﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.Schemas.Schemas
{
    /// <summary>
    /// This class is used as a base as a cut down version of the PatientMaster class
    /// </summary>
    ///
    [KnownType(typeof(PatientMasterBase))]
    [Serializable]
    [DataContract]
    public class PatientMasterBase : BaseSchema
    {
        /// <summary>
        /// Gets the patients current name.
        /// </summary>
        /// <value>
        /// The name of the current.
        /// </value>
        public PatientMasterName CurrentName
        {
            get
            {
                PatientMasterName currentName = GetName(NameTypes.Current);
                if (currentName == null)
                {
                    currentName = new PatientMasterName();
                    currentName.NameTypeId = (int)NameTypes.Current;
                }
                return currentName;
            }
        }

        /// <summary>
        /// Gets or sets the current sex id.
        /// </summary>
        /// <value>
        /// The current sex id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int CurrentSexId { get; set; }

        /// <summary>
        /// Gets or sets the date of birth.
        /// </summary>
        /// <value>
        /// The date of birth.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets the date of death.
        /// </summary>
        /// <value>
        /// The date of death.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public DateTime? DateOfDeath { get; set; }

        /// <summary>
        /// Gets or sets the names.
        /// </summary>
        /// <value>
        /// The names.
        /// </value>
        [DataMember]
        public List<PatientMasterName> Names { get; set; }

        /// <summary>
        /// Gets or sets the patient master ID.
        /// </summary>
        /// <value>
        /// The patient master ID.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.AutoGeneratedKey)]
        public int? PatientMasterId
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }

        #region Methods

        /// <summary>
        /// Sets the new name.
        /// </summary>
        /// <param name="titleId">The title id.</param>
        /// <param name="firstNames">The first names.</param>
        /// <param name="lastName">The last name.</param>
        /// <param name="suffixId">The suffix id.</param>
        public void SetNewCurrentName(int? titleId, string firstNames, string lastName, int? suffixId)
        {
            int nameTypeId = (int)NameTypes.Current;
            PatientMasterName name = new PatientMasterName(titleId, null, firstNames, lastName, suffixId, null, nameTypeId, null);
            PatientMasterName existingName = this.Names.Find(result => (result.CompareKey == name.CompareKey));
            PatientMasterName oldCurrent = Names.Find(result => result.NameTypeId == nameTypeId);
            if (existingName != null)
            {
                existingName.NameTypeId = nameTypeId;
            }
            else
            {
                Names.Add(name);
            }

            if ((oldCurrent != null) && (oldCurrent.CompareKey != name.CompareKey))
            {
                oldCurrent.NameTypeId = (int)NameTypes.Previous;
                oldCurrent.NameType = string.Empty;
            }
        }

        /// <summary>
        /// Gets the name.
        /// </summary>
        /// <param name="type">The type.</param>
        /// <returns></returns>
        private PatientMasterName GetName(NameTypes type)
        {
            return Names.FirstOrDefault(result => result.NameTypeId == (int)type);
        }

        #endregion Methods
    }
}