﻿using System;
using System.Runtime.Serialization;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.PcehrSchemas
{
    /// <summary>
    /// This class represents a queued operation to upload a document to a PCEHR.
    /// </summary>
    [KnownType(typeof(QueuedUploadOperation))]
    [Serializable]
    [DataContract]
    public class QueuedUploadOperation
    {
        #region Private members

        private Participant author;
        private DocumentFormat documentFormat;
        private DocumentType documentType;
        private Episode episode;
        private Hospital hospital;
        private HospitalPatient hospitalPatient;
        private byte[] package;
        private PatientIdentifierBase patientIdentifier;
        private PatientMaster patientMaster;
        private PcehrMessageQueue pendingItem;
        private string sourceSystemDocumentId;
        private string sourceSystemSetId;
        private UserDetails user;

        #endregion Private members

        #region properties

        /// <summary>
        /// The document author.
        /// </summary>
        [DataMember]
        public Participant Author
        {
            get
            {
                return this.author;
            }
            set
            {
                this.author = value;
            }
        }

        /// <summary>
        /// The document format. This is not a fixed enumerated list, but a
        /// reference to a record in the DocumentFormat table in the database.
        /// Additional document formats can be configured in this table. They
        /// are looked up by the document format code specified in the
        /// parameter on the upload document call. If the parameter is not
        /// given then the application setting DefaultDocumentFormatCode takes
        /// effect.
        /// </summary>
        [DataMember]
        public DocumentFormat DocumentFormat
        {
            get
            {
                return documentFormat;
            }
            set
            {
                this.documentFormat = value;
            }
        }

        /// <summary>
        /// The document type. This is not a fixed enumerated list, but a reference to a record
        /// in the DocumentType table in the database. Additional document types can be configured
        /// in this table. They are looked up by the top-level code element in the CDA document.
        /// </summary>
        [DataMember]
        public DocumentType DocumentType
        {
            get
            {
                return documentType;
            }
            set
            {
                this.documentType = value;
            }
        }

        /// <summary>
        /// The episode of care that this document was written about.
        /// </summary>
        [DataMember]
        public Episode Episode
        {
            get
            {
                return this.episode;
            }
            set
            {
                this.episode = value;
            }
        }

        /// <summary>
        /// The hospital in which the service that led to the creation of the document occurred.
        /// </summary>
        [DataMember]
        public Hospital Hospital
        {
            get
            {
                return this.hospital;
            }
            set
            {
                this.hospital = value;
            }
        }

        /// <summary>
        /// The patient's identity within the hospital.
        /// </summary>
        [DataMember]
        public HospitalPatient HospitalPatient
        {
            get
            {
                return this.hospitalPatient;
            }
            set
            {
                this.hospitalPatient = value;
            }
        }

        /// <summary>
        /// The CDA package.
        /// </summary>
        [DataMember]
        public byte[] Package
        {
            get
            {
                return this.package;
            }
            set
            {
                this.package = value;
            }
        }

        /// <summary>
        /// The patient identifier.
        /// </summary>
        [DataMember]
        public PatientIdentifierBase PatientIdentifier
        {
            get
            {
                return this.patientIdentifier;
            }
            set
            {
                this.patientIdentifier = value;
            }
        }

        /// <summary>
        /// The patient, being the subject of care, about which the document was written.
        /// </summary>
        [DataMember]
        public PatientMaster PatientMaster
        {
            get
            {
                return this.patientMaster;
            }
            set
            {
                this.patientMaster = value;
            }
        }

        /// <summary>
        /// The pending item in the PcehrMessageQueue table in the database.
        /// </summary>
        [DataMember]
        public PcehrMessageQueue PendingItem
        {
            get
            {
                return this.pendingItem;
            }
            set
            {
                this.pendingItem = value;
            }
        }

        /// <summary>
        /// The source system document identifier, which can be root^extension or just root.
        /// This identifier must be globally unique, and different between versions of the
        /// same document.
        /// </summary>
        [DataMember]
        public string SourceSystemDocumentId
        {
            get
            {
                return this.sourceSystemDocumentId;
            }
            set
            {
                this.sourceSystemDocumentId = value;
            }
        }

        /// <summary>
        /// The source system set ID, which can be an OID or OID^extension.
        /// This identifier must be the same between versions of the same document,
        /// but globally unique as an identifier of the document.
        /// </summary>
        [DataMember]
        public string SourceSystemSetId
        {
            get
            {
                return this.sourceSystemSetId;
            }
            set
            {
                this.sourceSystemSetId = value;
            }
        }

        /// <summary>
        /// The system user who initiated the upload of the document to the PCEHR.
        /// This need not be the same person as the document's author or legal authenticator.
        /// </summary>
        [DataMember]
        public UserDetails User
        {
            get
            {
                return this.user;
            }
            set
            {
                this.user = value;
            }
        }

        #endregion properties

        #region constructor

        /// <summary>
        /// Initializes a new instance of the <see cref="QueuedUploadOperation" /> class.
        /// </summary>
        /// <param name="user">The user.</param>
        public QueuedUploadOperation(UserDetails user)
        {
            this.user = user;
        }

        #endregion constructor
    }
}