﻿using System;
using System.Globalization;
using System.Linq;

namespace Test.CommonCcaNoc.Helpers
{
    /// <summary>
    /// Contains extension methods.
    /// </summary>
    public static class Extensions
    {
        /// <summary>
        /// Extension Methods for retrieving a value from an attribute
        /// </summary>
        /// <typeparam name="T">The object / attribute type</typeparam>
        /// <typeparam name="TExpected">The expected return value type</typeparam>
        /// <param name="enumeration">The enum type that this method extends</param>
        /// <param name="expression">An expression specifying the property on the attribute you would like to retrun</param>
        /// <returns>The value as specified by the expression parameter</returns>
        public static TExpected GetAttributeValue<T, TExpected>
            (
                this Enum enumeration,
                Func<T, TExpected> expression
            )
            where T : Attribute
        {
            var attribute = enumeration.GetType().GetMember(enumeration.ToString())[0].GetCustomAttributes(typeof(T), false).Cast<T>().SingleOrDefault();

            return attribute == null ? default(TExpected) : expression(attribute);
        }

        /// <summary>
        /// Converts an HL7 timestamp string to Universal Coordinated Time (UTC).
        /// </summary>
        /// <param name="timestring">The timestamp string.</param>
        /// <returns>The timestamp in UTC.</returns>
        public static string ToUtc(this string timestring)
        {
            var formatTemplate = "yyyyMMddHHmmss.ffff";

            var format = string.Empty;
            var timezoneFormat = string.Empty;

            int plusMinusIndex = timestring.IndexOf("+");
            if (plusMinusIndex < 0) plusMinusIndex = timestring.IndexOf("-");

            if (plusMinusIndex >= 10)
            {
                format = formatTemplate.Substring(0, plusMinusIndex);
                int timezoneLength = timestring.Substring(plusMinusIndex + 1).Length;

                if (timezoneLength != 2 && timezoneLength != 4)
                    throw new FormatException("Timezone must be specified with either 2 or 4 digits.");

                timezoneFormat = "zzzz".Substring(0, timezoneLength);
                var equivalent = DateTime.ParseExact(timestring, format + timezoneFormat, CultureInfo.InvariantCulture);

                // Get output format
                string outputFormat = format;
                if (timezoneLength == 4 && !timestring.EndsWith("00") && format.Length < 12)
                    outputFormat = formatTemplate.Substring(0, 12);
                if (outputFormat.Length > 14)
                    outputFormat = outputFormat.Substring(0, 14);

                return equivalent.ToUniversalTime().ToString(outputFormat);
            }
            else if (plusMinusIndex < 10 && plusMinusIndex > -1)
            {
                return timestring.Substring(0, plusMinusIndex);
            }
            else
            {
                if (timestring.Length > 14)
                    return timestring.Substring(0, 14);
                else
                    return timestring;
            }
        }
    }
}