﻿using System;
using System.Linq;
using System.Collections.Generic;
using System.Threading;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using HIPS.ServiceContracts.Pcehr.Message;
using HIPS.ServiceContracts.Common.DTO.UserIdentity;
using HIPS.ServiceContracts.Common.DTO.ParticipatingIndividual;
using HIPS.ServiceContracts.Common.DTO.PatientIdentifier;
using HIPS.ServiceContracts.Common.DTO;
using HIPS.ServiceContracts.Common;

namespace Test.CommonCcaNoc.Helpers
{
    /// <summary>
    /// This class contains helpers for tests involving the MSMQ queue for PCEHR Upload/Remove.
    /// </summary>
    public class QueueHelper
    {
        /// <summary>
        /// Calls the HIPS service UploadOrSupersedeDocument and waits until
        /// there is no pending operation on the queue, or else until the timeout
        /// specified in the app.config file has expired.
        /// </summary>
        /// <param name="cdaDocument"></param>
        public static void UploadDocumentAndWaitUntilProcessed(CcaPatient patient, LogAssert assert, CdaDocument cdaDocument)
        {
            byte[] contents = cdaDocument.GetBytes();
            string formatCode = cdaDocument.GetFormatCode();
            DateTime admission = patient.TargetEpisode.AdmissionDate;
            HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase id = patient.TargetPatientIdentifier;
            UserDetails user = patient.GetTestUser();
            HipsResponse response = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(contents, id, user, new Attachment[0], admission, formatCode);
            assert.ExpectResponse(HipsResponseIndicator.OK, response, DialogueResource.HipsServiceUploadDocument);
            WaitForQueuedOperation(patient, assert, DialogueResource.UploadWaitTimeout);
        }

        /// <summary>
        /// Calls the HIPS service UploadOrSupersedeDocument and waits until
        /// there is no pending operation on the queue, or else until the timeout
        /// specified in the app.config file has expired.
        /// </summary>
        /// <param name="cdaDocument"></param>
        public static void UploadDocument1AAndWaitUntilProcessed(CcaPatient patient, LogAssert assert, byte[] pdf)
        {
            DateTime admission = patient.TargetEpisode.AdmissionDate;
            LocalUser user = patient.GetDTOLocalTestUser();

            ParticipatingProvider provider = CcaPatient.ParticipatingProvider;

            //ParticipatingProvider provider = new ParticipatingProvider()
            //{
            //    FamilyName = "SANDEEN",
            //    GivenNames = "MILO",
            //    Hpii = "8003619900000081",
            //    Title = null,
            //    Suffix = null,
            //    LocalIdentifier = null
            //};

            HIPS.ServiceContracts.Common.DTO.PatientIdentifier.Mrn patientIdentifier = new HIPS.ServiceContracts.Common.DTO.PatientIdentifier.Mrn()
            {
                Value = patient.TargetHospitalPatient.Mrn,
                HospitalCode = patient.TargetPatientIdentifier.HospitalCode,
                HospitalCodeSystem = patient.TargetPatientIdentifier.HospitalCodeSystem
            };

            CdaHeaderMetadata metadata = new CdaHeaderMetadata()
            {
                AdmissionDateTime = admission,
                DischargeDateTime = admission.AddDays(1),
                DocumentAuthor = provider,
                DocumentCreationDateTime = DateTime.Now,
                LegalAuthenticator = provider,
                ModeOfSeparation = ModeOfSeparation.Home,
                ResponsibleHealthProfessional = provider,
                SourceDocumentStatus = SourceDocumentStatus.Interim,
                Specialty = "General Practitioner"
            };

            CdaAttachment[] attachments = new CdaAttachment[0];

            UploadDischargeSummaryLevel1ARequest request =  new UploadDischargeSummaryLevel1ARequest
                                                            {
                                                                User = user,
                                                                PdfDocument = pdf,
                                                                PatientIdentifier = patientIdentifier,
                                                                CdaHeaderMetadata = metadata,
                                                                Attachments = attachments.ToList()
                                                            };
            

            UploadDischargeSummaryLevel1AResponse response = ProxyHelper.PcehrServiceV2Proxy.UploadOrSupersedeDischargeSummaryLevel1A(request);
            assert.ExpectResponseStatus(ResponseStatus.OK, response.Status, DialogueResource.HipsServiceUploadDocument);
            WaitForQueuedOperation(patient, assert, DialogueResource.UploadWaitTimeout);
        }

        /// <summary>
        /// Waits until the upload or removal has gone through the queue.
        /// </summary>
        /// <param name="patient">The CCA test patient</param>
        /// <param name="assert">The test assertion helper</param>
        /// <param name="timeoutMessage">The message to log if the operation does not complete within the configured timeout</param>
        /// <param name="expectFailure">Whether to expect failure (optional; default is false, to expect success)</param>
        public static void WaitForQueuedOperation(CcaPatient patient, LogAssert assert, string timeoutMessage, bool expectFailure = false)
        {
            if (!patient.TargetEpisode.EpisodeId.HasValue)
            {
                patient.GetDatabaseIds(needEpisode: true);
            }

            // Wait until the removal has gone through the queue.
            int i = 0;
            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            while (queueOps.Count > 0 && queueOps[0].QueueStatusId == (int)QueueStatus.Pending)
            {
                if (i++ > CcaPatient.PcehrQueueTimeout)
                {
                    assert.Fail(timeoutMessage);
                }
                Thread.Sleep(1000);
                queueOps = patient.GetPcehrMessageQueueItems();
            }

            if (expectFailure)
            {
                if (queueOps.Count == 0)
                {
                    assert.Fail(DialogueResource.QueuedOperationSuccessful);
                }
                assert.AreEqual(QueueStatus.Failure, (QueueStatus)queueOps[0].QueueStatusId, DialogueResource.QueueOperationStatus);
            }
            else
            {
                // Check that the queued operation was a success
                if (queueOps.Count > 0)
                {
                    assert.AreEqual(QueueStatus.Success, (QueueStatus)queueOps[0].QueueStatusId, DialogueResource.QueueOperationStatus);
                }
                else
                {
                    // Nothing was returned - assume that the queued operation was deleted on success
                }
            }
        }
    }
}