﻿using System;
using System.IO;
using System.Text;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentProduction
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  For All Document Exchange Services
    /// </summary>
    [TestClass]
    public class NOC_AllDocumentExchangeServices : CcaTest
    {
        /// <summary>
        /// Test No:         55
        /// Objective/Input:
        ///
        /// Ensure the client system includes PCEHR Common Header Elements and
        /// Transmission Signature containing a signed attestation of elements
        /// contained within the SOAP message on all SOAP Requests.
        ///
        ///   - For this test case, the client system uses valid HPI-O
        ///     certificates provided in the test data package. Service Invoker
        ///     and Service Provider SHALL establish a secure communication
        ///     channel when interacting with each other.
        ///   - Implementation confirms to web services base profile and TLS
        ///     security profile.
        ///   - Implementation  does not rely on support for IHE interactions.
        ///   - Unique message ID  is used in all transactions.
        ///   - IHI number is specified in all transactions, is valid, matches
        ///     the individual who owns the PCEHR and is numerical.
        ///   - ID is specified in all transactions, has no leading or trailing
        ///     spaces. If ID is HPII, ensure it is valid and numerical.
        ///   - If role and/or alternate organisation is supplied, no leading
        ///     or trailing spaces used
        ///   - User name, vendor, product name, production version, platform,
        ///     organisation ID, organisation name,  has no leading or trailing
        ///     space, it is not blank.
        ///   - Organisation name is set to the name of the organisation.
        ///   - Product name, product version, product type and product
        ///     platform are set to the appropriate values as per client
        ///     system.
        ///
        /// Expected Result:
        ///
        /// The requests are transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_055()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            PcehrHeaderTests headerTests = new PcehrHeaderTests(this);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary3A);
            headerTests.CheckHeader(GetUploadDocumentRequest(document),
                DialogueResource.HipsServiceUploadDocument);
            headerTests.CheckHeader(GetRetrieveDocumentRequest(document),
                DialogueResource.HipsServiceRetrieveDocument);
            headerTests.CheckHeader(GetRemoveDocumentRequest(document),
                DialogueResource.HipsServiceRemove);
        }

        /// <summary>
        /// Removes the document from the PCEHR system and returns the SOAP
        /// request that was submitted to the PCEHR system to remove the
        /// document.
        /// </summary>
        /// <param name="document">The document that is to be removed.</param>
        /// <returns>The SOAP request</returns>
        private XmlDocument GetRemoveDocumentRequest(CdaDocument document)
        {
            byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("NOC TEST 55");
            HipsResponse response = ProxyHelper.PcehrProxy.Remove(
                patient.TargetPatientIdentifier,
                patient.TargetEpisode.AdmissionDate,
                document.GetSetId(),
                RemovalReason.Withdrawn,
                patient.GetTestUser(),
                auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout);
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.RemoveDocument, audit.ServiceName, DialogueResource.PcehrAuditServiceName);
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }

        /// <summary>
        /// Finds the document using a query by document unique ID and then
        /// retrieves the document from the PCEHR system, and returns the
        /// SOAP request used to retrieve the document.
        /// </summary>
        /// <param name="document">The document to retrieve</param>
        /// <returns>The SOAP request to the retrieve document service</returns>
        private XmlDocument GetRetrieveDocumentRequest(CdaDocument document)
        {
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            DocumentQuery query = new DocumentQuery() { DocumentUniqueId = document.GetDocId() };
            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentList(identifier, user, query);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListWithQuery);
            DocumentRequest request = new DocumentRequest();
            request.DocumentUniqueId = listResponse.DocumentList[0].DocumentUniqueId;
            request.RepositoryUniqueId = listResponse.DocumentList[0].RepositoryUniqueId;
            DocumentResponse response = ProxyHelper.PcehrProxy.RetrieveDocument(identifier, user, request);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response.HipsResponse, DialogueResource.HipsServiceRetrieveDocument);
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.RetrieveDocument, audit.ServiceName, DialogueResource.PcehrAuditServiceName);
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }

        /// <summary>
        /// Uploads the document to the PCEHR and returns the SOAP request
        /// that was submitted to the PCEHR system.
        /// </summary>
        /// <param name="document">The document to upload</param>
        /// <returns>The SOAP request</returns>
        private XmlDocument GetUploadDocumentRequest(CdaDocument document)
        {
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.UploadDocument, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }
    }
}