﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.NehtaCca.Helpers;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.CommonSchemas.PatientIdentifier;
using System.Xml;
using System.IO;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.DataAccess;
using System.Data.SqlClient;

namespace Test.NehtaCca.NoticeOfConnection.GetChangeHistoryView
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  getChangeHistoryView
    /// </summary>
    [TestClass]
    public class NOC_GetChangeHistoryView : CcaTest
    {
        /// <summary>
        /// Test No:         6
        /// Objective/Input: Ensure the DocumentID is used for the document unique
        ///                  identifier which is requested for the document change
        ///                  history view operation.
        /// Expected Result: The request is transmitted successfully. Confirmation
        ///                  that the payload is well formed and complies with the
        ///                  getChangeHistoryView Request XML Schema. The document
        ///                  id is provided.
        /// </summary>
        [TestMethod]
        public void PCEHR_NOC_006()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            // This is Test Document ID 106 in the Conformance Test Data for CIS connecting to PCEHR v1.8
            string documentId = CdaDocument.ConvertToOid("ebf96c6f-3252-47b0-aa89-b72740d5672d");

            // Get the list of historical versions and verify that there are 4 versions and all are event summaries
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(identifier, user, documentId);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual(4, historyResponse.DocumentList.Count, DialogueResource.DocumentListCount);
            foreach (DocumentMetaDataItem item in historyResponse.DocumentList)
            {
                LogAssert.AreEqual("Event Summary", historyResponse.DocumentList[0].DocumentClassName, DialogueResource.DocumentClassName);
            }

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetChangeHistoryView, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Confirm that the payload is well formed and complies with the
            /// getChangeHistoryView Request XML Schema.
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/GetChangeHistoryView/1.0");
            XmlNode payload = doc.SelectSingleNode("/s:Envelope/s:Body/p:getChangeHistoryView", nsmgr);
            doc.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("p"), "PCEHR_GetChangeHistoryView.xsd");
            try
            {
                doc.Validate(null, payload);
                LogAssert.IsTrue(true, "Validates against getChangeHistoryView schema", "no");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }

            // Check that the document ID was provided in the request
            XmlNode documentIdNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:getChangeHistoryView/documentID", nsmgr);
            LogAssert.AreEqual(documentId, documentIdNode.InnerText, "Document ID Node value");
        }

        /// <summary>
        /// Test No:         7
        /// Objective/Input: Verify that the client system can handle a Functional Error (PCEHR_ERROR_3002 - Document Metadata Failed Validation) when a get change history view operation is performed for a document that has been deleted.
        /// Expected Result: The request is transmitted successfully and a ( PCEHR_ERROR_3002 - Document Metadata Failed Validation ) is received. The client system is able to handle the error and inform the user appropriately. 
        /// </summary>
        [TestMethod]
        public void PCEHR_NOC_007()
        {
            // Create and upload the initial version of the document
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);
            CdaDocument cdaDocument = patient.CreateNewDocument();
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, cdaDocument.GetBytes());

            // Perform an operation to remove the document from the PCEHR System.
            byte[] auditInformation = Encoding.UTF8.GetBytes("PCEHR_NOC_7");
            HipsResponse response = ProxyHelper.PcehrProxy.Remove(patient.TargetPatientIdentifier, patient.TargetEpisode.AdmissionDate, cdaDocument.GetSetId(), RemovalReason.Withdrawn, patient.GetTestUser(), auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout);

            // Gain emergency access to the PCEHR
            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(patient.TargetPatientIdentifier,  patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            // Perform a get change history view operation for the document that has just been deleted
            string documentId = cdaDocument.GetDocId();
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(patient.TargetPatientIdentifier,  patient.GetTestUser(), documentId);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);

            // Check the client system is able to handle the error and inform the user appropriately. 
            Log(DialogueResource.InformingUserIsOutOfScope);
        }

        /// <summary>
        /// Test No:         8
        /// Objective/Input: To show the ability of the client system to handle 
        ///                  a Standard Error (PCEHR_ERROR_0505 - Invalid HPI-O)
        ///                  when the user performs a  get change history view
        ///                  operation. The client system is configured with an
        ///                  invalid HPI-O. 
        /// Expected Result: The request is transmitted successfully to the PCEHR
        ///                  System. A SOAP Fault with Standard Error
        ///                  PCEHR_ERROR_0505 - Invalid HPI-O is received. Client
        ///                  system is able to handle and inform the user
        ///                  regarding the error.
        /// </summary>
        [TestMethod]
        public void PCEHR_NOC_008()
        {
            // Create the patient record
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);
            
            // Modify the HPI-O
            SqlCommand command = patient.TestCommand;
            command.CommandText = "UPDATE hips.HealthProviderOrganisation SET HpiO='8003623233353556' WHERE HealthProviderOrganisationId = 1";
            command.ExecuteNonQuery();
            HipsResponse reloadResponse = ProxyHelper.ReferenceProxy.ReloadReferenceData();
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, reloadResponse, DialogueResource.HipsServiceReloadReferenceData);

            string documentId = "1.2.3.4.5";
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(patient.TargetPatientIdentifier, patient.GetTestUser(), documentId);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual("PCEHR_ERROR_0505 - Invalid HPI-O", historyResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);

            // Check the client system is able to handle the error and inform the user appropriately. 
            Log(DialogueResource.InformingUserIsOutOfScope);

            // Restore the correct HPI-O
            SqlCommand command2 = patient.TestCommand;
            command.CommandText = "UPDATE hips.HealthProviderOrganisation SET HpiO='8003623233353555' WHERE HealthProviderOrganisationId = 1";
            command.ExecuteNonQuery();
            HipsResponse reloadResponse2 = ProxyHelper.ReferenceProxy.ReloadReferenceData();
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, reloadResponse2, DialogueResource.HipsServiceReloadReferenceData);
        }
    }
}
