﻿using System.Collections.Generic;
using System.Security.Cryptography;
using System.Security.Cryptography.Xml;
using System.Xml;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection
{
    /// <summary>
    /// Performs common tests on the PCEHR headers.
    /// This is used across multiple NOC tests.
    /// </summary>
    public class PcehrHeaderTests
    {
        private Dictionary<string, string> allMessageIds = new Dictionary<string, string>();
        private XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
        private LogAssert logAssert;
        private CcaPatient patient;
        private CcaTest test;

        /// <summary>
        /// Initialises the test details and XML namespaces for checking a PCEHR header.
        /// </summary>
        /// <param name="test">The current NOC test</param>
        public PcehrHeaderTests(CcaTest test)
        {
            this.test = test;
            this.patient = test.patient;
            this.logAssert = test.LogAssert;
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("a", "http://www.w3.org/2005/08/addressing");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/PCEHRProfile/1.0");
            nsmgr.AddNamespace("h", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
            nsmgr.AddNamespace("d", "http://www.w3.org/2000/09/xmldsig#");
            nsmgr.AddNamespace("mssmd", "http://schemas.microsoft.com/2004/09/ServiceModel/Diagnostics");
            nsmgr.AddNamespace("mssms", "http://schemas.microsoft.com/vstudio/diagnostics/servicemodelsink");
        }

        /// <summary>
        /// Checks the PCEHR header for a particular SOAP request.
        /// </summary>
        /// <param name="soapRequest">The SOAP request to the PCEHR B2B Gateway</param>
        /// <param name="serviceName">The name of the HIPS service that was called (for logging)</param>
        public void CheckHeader(XmlDocument soapRequest, string serviceName)
        {
            test.Log("Now checking header for service {0}", serviceName);
            CheckMessageId(soapRequest, serviceName);
            CheckIhiNumber(soapRequest);
            CheckUser(soapRequest);
            CheckOrganisation(soapRequest);
            CheckProductType(soapRequest);
            CheckSignature(soapRequest);
            CheckNoVisualStudioAdditions(soapRequest);
        }

        /// <summary>
        /// Verify that the SOAP request does not contain any Microsoft diagnostic headers.
        /// </summary>
        /// <param name="soapRequest"></param>
        private void CheckNoVisualStudioAdditions(XmlDocument soapRequest)
        {
            XmlNode activityIdNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/mssmd:ActivityId", nsmgr);
            logAssert.IsTrue(activityIdNode == null, "There is no ActivityId element in the SOAP header", "There is an ActivityId element in the SOAP header");

            XmlNode vsDebuggerCausalityDataNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/mssms:VsDebuggerCausalityData", nsmgr);
            logAssert.IsTrue(vsDebuggerCausalityDataNode == null, "There is no VsDebuggerCausalityData element in the SOAP header", "There is a VsDebuggerCausalityData element in the SOAP header");
        }

        /// <summary>
        /// Verify that the SOAP request is signed with a valid signature.
        /// This tests that the client system includes a Transmission Signature containing a signed
        /// attestation of elements contained within the SOAP message on all SOAP Requests.
        /// </summary>
        private void CheckSignature(XmlDocument soapRequest)
        {
            SignedXml xml = new SignedXmlForPcehr(soapRequest);
            XmlNode signatureNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:signature/d:Signature", nsmgr);
            xml.LoadXml(signatureNode as XmlElement);

            try
            {
                bool isSignatureValid = xml.CheckSignature();
                logAssert.IsTrue(isSignatureValid, "SOAP request signature is Valid", "SOAP request signature is invalid");
            }
            catch (CryptographicException ex)
            {
                logAssert.Fail(ex.Message);
            }
        }

        /// <summary>
        /// Checks that the vendor, product name, product version and platform have no leading or trailing space and are not blank.
        /// </summary>
        /// <param name="soapRequest">The SOAP request</param>
        private void CheckProductType(XmlDocument soapRequest)
        {
            XmlNode vendorNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:productType/h:vendor", nsmgr);
            XmlNode productNameNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:productType/h:productName", nsmgr);
            XmlNode productVersionNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:productType/h:productVersion", nsmgr);
            XmlNode platformNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:productType/h:platform", nsmgr);

            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(vendorNode.InnerText, "Vendor");
            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(productNameNode.InnerText, "Product Name");
            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(productVersionNode.InnerText, "Product Version");
            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(platformNode.InnerText, "Platform");
        }

        /// <summary>
        /// Performs tests on the organisation details in the PCEHR header.
        /// Checks that the organisation ID and organisation name have no leading or trailing space and are not blank.
        /// If alternate organisation name is supplied, checks that no leading or trailing spaces are used.
        /// Checks that the organisation name is set to the name of the organisation.
        /// </summary>
        /// <param name="soapRequest">The SOAP request</param>
        private void CheckOrganisation(XmlDocument soapRequest)
        {
            XmlNode organisationIdNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:accessingOrganisation/h:organisationID", nsmgr);
            XmlNode organisationNameNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:accessingOrganisation/h:organisationName", nsmgr);
            XmlNode alternateOrganisationNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:accessingOrganisation/h:alternateOrganisationName", nsmgr);

            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(organisationIdNode.InnerText, "Organisation ID");
            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(organisationNameNode.InnerText, "Organisation Name");
            logAssert.AreEqual(patient.TargetHospital.HpioName, organisationNameNode.InnerText, "The name of the healthcare provider organisation", "the organisation name in the PCEHR header");

            if (alternateOrganisationNode != null)
            {
                logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(alternateOrganisationNode.InnerText, "Alternate Organisation Name");
            }
            else
            {
                test.Log("Info: Alternate Organisation Name was not supplied");
            }
        }

        /// <summary>
        /// Performs tests on the user details in the PCEHR header.
        /// Checks user ID is specified in all transactions, and has no leading or trailing spaces.
        /// If ID is HPII, ensures it is valid and numerical.
        /// Checks that the user name has no leading or trailing space and is not blank.
        /// If role is supplied, ensures that no leading or trailing spaces are used.
        /// </summary>
        /// <param name="soapRequest"></param>
        private void CheckUser(XmlDocument soapRequest)
        {
            XmlNode userNameNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:User/h:userName", nsmgr);
            XmlNode userIdNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:User/h:ID", nsmgr);
            XmlNode userIdTypeNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:User/h:IDType", nsmgr);
            XmlNode roleNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:User/h:role", nsmgr);

            logAssert.IsNotNull(userIdNode, "User ID Node");
            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(userIdNode.InnerText, "User ID");
            logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(userNameNode.InnerText, "User Name");

            if (roleNode != null)
            {
                logAssert.HasNoLeadingOrTrailingSpacesAndIsNotBlank(roleNode.InnerText, "Role");
            }
            else
            {
                test.Log("Info: Role was not supplied");
            }

            if (userIdTypeNode.InnerText == "HPII")
            {
                logAssert.IsNumerical(userIdNode.InnerText, "User ID");
            }
            else
            {
                test.Log("Info: User ID type was {0}, not HPII", userIdTypeNode.InnerText);
            }
        }

        /// <summary>
        /// Check IHI number is specified in all transactions, is valid, matches the individual who owns the PCEHR and is numerical.
        /// </summary>
        /// <param name="soapRequest"></param>
        private void CheckIhiNumber(XmlDocument soapRequest)
        {
            XmlNode ihiNumberNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/h:PCEHRHeader/h:ihiNumber", nsmgr);
            logAssert.IsNotNull(ihiNumberNode, "IHI element in the header");
            logAssert.IsNumerical(ihiNumberNode.InnerText, "IHI in the header");
            logAssert.AreEqual(patient.IhiInformation.Ihi, ihiNumberNode.InnerText, "IHI of the individual who owns the PCEHR", "IHI in the header");
        }

        /// <summary>
        /// Check Unique message ID  is used in all transactions.
        /// </summary>
        /// <param name="soapRequest"></param>
        /// <param name="serviceName"></param>
        private void CheckMessageId(XmlDocument soapRequest, string serviceName)
        {
            XmlNode messageIdNode = soapRequest.SelectSingleNode("/s:Envelope/s:Header/a:MessageID", nsmgr);
            bool isUnique = !allMessageIds.ContainsKey(messageIdNode.InnerText);
            string previouslyUsedFor = isUnique ? "Not previously used" : allMessageIds[messageIdNode.InnerText];
            logAssert.IsTrue(isUnique,
                string.Format("Message ID {0} is unique so far.", messageIdNode.InnerText),
                string.Format("Message ID {0} was used for both {1} and {2}.", messageIdNode.InnerText, serviceName, previouslyUsedFor));
            allMessageIds[messageIdNode.InnerText] = serviceName;
        }
    }
}