﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.IO;
using System.ServiceModel.Channels;
using System.ServiceModel.Description;
using System.ServiceModel.Dispatcher;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using HIPS.Client.Proxy;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.Registration
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  RegisterPCHER
    /// </summary>
    [TestClass]
    public class NOC_RegisterPcehr : CcaTest
    {
        /// <summary>
        /// Test No:         80
        /// Objective/Input:
        ///
        /// Perform a RegisterPCEHR operation with valid input elements.
        /// Ensure the individual IHI Number is used in the PCEHRHeader when the individual demographics are not used.
        ///
        /// IvcCorrespondenceChannel set to SMS
        /// 
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_080()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 22");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.sms;
            channel.phoneNumber = ConfigurationManager.AppSettings["IvcSmsPhoneNumber"].ToString();

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that the IHI number was included in the request
            XmlDocument xmlRequest;
            XmlNamespaceManager nsmgr;

            GetAuditRequest("Register PCEHR", out xmlRequest, out nsmgr);

            XmlNode ihiNumberNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.AreEqual(patient.IhiInformation.Ihi, ihiNumberNode.InnerText, "IHINumber element value");
        }

        /// <summary>
        /// Test No:         81
        /// Objective/Input:
        ///
        /// To show the ability of the client system to use the individual demographics
        /// when an IHI number is not provided in the PCEHRHeader.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS)
        /// 
        /// 
        /// ************************************************************************************************************
        /// MUST CHANAGE THE "BypassHIService" configuration item to True to pass this test in Running HIPS
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_081()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "AR ID 04", preloadPatient: false);

            // Create a demographic which contains no IHI
            Demographic demographics = patient.TargetDemographic;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(demographics,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that the IHI number was not included in the request
            XmlDocument xmlRequest;
            XmlNamespaceManager nsmgr;

            GetAuditRequest("Register PCEHR", out xmlRequest, out nsmgr);

            // Ensure the IHI header value is empty
            XmlNode ihiNumberNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.IsTrue(ihiNumberNode == null, "No IHI number recorded", "IHI number found");

            // Ensure the patient demographics were included
            XmlNode repFamilyNameNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:name/c:familyName", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.FamilyName, repFamilyNameNode.InnerText, "Family Name element value");
            XmlNode repGivenNameNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:name/c:givenName", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.GivenName, repGivenNameNode.InnerText, "Given Name element value");
            XmlNode repSexNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:sex", nsmgr);
            LogAssert.AreEqual(AuditHelper.GetSex(patient.TargetDemographic.Sex), repSexNode.InnerText, "Sex element value");
            XmlNode repDateOfBirthNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:dateOfBirth", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.DateOfBirth, DateTime.Parse(repDateOfBirthNode.InnerText), "Date of Birth element value");

            Log("Used PatientDemographics Object: {0}", demographics.GetType().Name);
            Log("-->FamilyName: {0}", demographics.FamilyName);
            Log("-->GivenName: {0}", demographics.GivenName);
            Log("-->DOB: {0}", demographics.DateOfBirth);
            Log("-->Sex: {0}", demographics.Sex);
            Log("-->MedicareNumber: {0}", demographics.MedicareNumber);
            Log("-->MedicareIrn: {0}", demographics.MedicareIrn);
            Log("-->DVA: {0}", demographics.DvaNumber);
            Log("-->HospitalCode: {0}", demographics.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", demographics.HospitalCodeSystem);

        }

        /// <summary>
        /// Test No:         81
        /// Objective/Input:
        ///
        /// To show the ability of the client system to use the individual demographics
        /// when an IHI number is not provided in the PCEHRHeader.
        ///
        /// Expected Result:
        /// This is a modification of the Test 81.  This is a NEGATIVE TEST.
        /// 
        /// Passed in NO Medicare Card with a Validated IHI
        /// 
        /// 
        /// ************************************************************************************************************
        /// MUST CHANAGE THE "BypassHIService" configuration item to True to pass this test in Running HIPS
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_081A()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "AR ID 09", preloadPatient: true);

            // Create a demographic
            Demographic demographics = patient.TargetDemographic;

            // Remove the Medicare number and Irn
            demographics.MedicareNumber = string.Empty;
            demographics.MedicareIrn = string.Empty;


            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(demographics,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Patient: Must include a Medicare or DVA number.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);


            Log("Used PatientDemographics Object: {0}", demographics.GetType().Name);
            Log("-->FamilyName: {0}", demographics.FamilyName);
            Log("-->GivenName: {0}", demographics.GivenName);
            Log("-->DOB: {0}", demographics.DateOfBirth);
            Log("-->Sex: {0}", demographics.Sex);
            Log("-->MedicareNumber: {0}", demographics.MedicareNumber);
            Log("-->MedicareIrn: {0}", demographics.MedicareIrn);
            Log("-->DVA: {0}", demographics.DvaNumber);
            Log("-->HospitalCode: {0}", demographics.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", demographics.HospitalCodeSystem);

        }

        /// <summary>
        /// Test No:         81
        /// Objective/Input:
        ///
        /// ******************
        /// 
        /// This is not a test issues by NOC.
        /// It is an extra test that is used to test DVA with no Medicare, using demographics (no IHI).
        /// 
        /// ******************
        /// To show the ability of the client system to use the individual demographics
        /// when an IHI number is not provided in the PCEHRHeader.
        ///
        /// Expected Result:
        /// This is a modification of the Test 81.  This is a NEGATIVE TEST.  
        /// 
        /// Passed in NO Medicare Card with a DVA.  
        /// 
        /// 
        /// ************************************************************************************************************
        /// MUST CHANAGE THE "BypassHIService" configuration item to True to pass this test in Running HIPS
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_081B()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_38", preloadPatient: false);

            // Create a demographic
            Demographic demographics = patient.TargetDemographic;

            // Remove the Medicare number and Irn
            demographics.MedicareNumber = string.Empty;
            demographics.MedicareIrn = string.Empty;

            //add DVA
            demographics.DvaNumber = "N 234578J";


            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(demographics,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("PCEHR_ERROR_9008", registrationResponse.HipsResponse.ResponseCode, DialogueResource.HipsServiceRegisterPatient);

            Log("Used PatientDemographics Object: {0}", demographics.GetType().Name);
            Log("-->FamilyName: {0}", demographics.FamilyName);
            Log("-->GivenName: {0}", demographics.GivenName);
            Log("-->DOB: {0}", demographics.DateOfBirth);
            Log("-->Sex: {0}", demographics.Sex);
            Log("-->MedicareNumber: {0}", demographics.MedicareNumber);
            Log("-->MedicareIrn: {0}", demographics.MedicareIrn);
            Log("-->DVA: {0}", demographics.DvaNumber);
            Log("-->HospitalCode: {0}", demographics.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", demographics.HospitalCodeSystem);

        }
        /// <summary>
        /// Test No:         82
        /// Objective/Input:
        ///
        /// To show the ability of the client system to use the individual Medicare
        /// card Number, IRN Number and demographics when an IHI number is not
        /// provided in the PCEHRHeader.
        ///
        /// IvcCorrespondenceChannel set to Email
        /// 
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS)
        /// 
        /// ************************************************************************************************************
        /// MUST CHANAGE THE "BypassHIService" configuration item to True to pass this test.
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_082()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "AR ID 05", preloadPatient: false);

            // Create a demographic which contains no IHI
            Demographic demographics = patient.TargetDemographic;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.email;
            channel.email = ConfigurationManager.AppSettings["IvcEmailAddress"].ToString();

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(demographics,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that the IHI number was not included in the request
            XmlDocument xmlRequest;
            XmlNamespaceManager nsmgr;

            GetAuditRequest("Register PCEHR", out xmlRequest, out nsmgr);

            // Ensure the IHI header value is empty
            XmlNode ihiNumberNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.IsTrue(ihiNumberNode == null, "No IHI number recorded", "IHI number found");

            // Ensure the Medicare Number and IRN were included
            XmlNode repMedicareNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:medicareCardNumber", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.MedicareNumber, repMedicareNode.InnerText, "Medicare Card Number element value");
            XmlNode repMedicareIrnNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:medicareIRN", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.MedicareIrn, repMedicareIrnNode.InnerText, "Medicare IRN element value");
        }

        /// <summary>
        /// Test No:         83
        /// Objective/Input:
        ///
        /// Verify that when an individual IHI Number nor the individual demographics
        /// are provided in the PCEHRHeader, an error message appears and the request
        /// is not transmitted to PCEHR system.
        ///
        /// Expected Result:
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that an IHI Number for the individual MUST be supplied
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_083A()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "AR ID 05");

            ValidatedIhi patientValidatedIhi = new ValidatedIhi(string.Empty,
                IhiStatus.Unknown,
                IhiRecordStatus.Unknown,
                new DateTime(2014, 01, 22),
                "Family",
                "Given",
                new DateTime(1995, 12, 15),
                SexEnumerator.NotStatedOrInadequatelyDescribed,
                "RNH",
                "pasFacCd");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patientValidatedIhi,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            Log("Used PatientIdentifier Object: {0}", patientValidatedIhi.GetType().Name);
            Log("-->IHI: {0}", patientValidatedIhi.Ihi);
            Log("-->IhiStatus: {0}", patientValidatedIhi.IhiStatus);
            Log("-->IhiRecordStatus: {0}", patientValidatedIhi.IhiRecordStatus);
            Log("-->IhiLastValidated: {0}", patientValidatedIhi.IhiLastValidated);
            Log("-->FamilyName: {0}", patientValidatedIhi.FamilyName);
            Log("-->GivenName: {0}", patientValidatedIhi.GivenName);
            Log("-->DOB: {0}", patientValidatedIhi.DateOfBirth);
            Log("-->Sex: {0}", patientValidatedIhi.Sex);
            Log("-->HospitalCode: {0}", patientValidatedIhi.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", patientValidatedIhi.HospitalCodeSystem);


            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, registrationResponse.HipsResponse, "Missing IHI");
            //LogAssert.AreEqual("Individuals details must be provided.", registrationResponse.HipsResponse.HipsErrorMessage, "Incorrect validation");
            LogAssert.AreEqual("No IHI was found for this patient and so the PCEHR cannot be accessed.", registrationResponse.HipsResponse.HipsErrorMessage, "ValidationError");
        }

        /// <summary>
        /// Test No:         83
        /// Objective/Input:
        ///
        /// Verify that when an individual IHI Number nor the individual demographics
        /// are provided in the PCEHRHeader, an error message appears and the request
        /// is not transmitted to PCEHR system.
        ///
        /// Expected Result:
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that an IHI Number for the individual MUST be supplied
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_083B()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "AR ID 05");

            Demographic patientDemographic = new Demographic(string.Empty,
                "Test Patient First Name",
                DateTime.MinValue,
                SexEnumerator.NotStatedOrInadequatelyDescribed,
                string.Empty,
                string.Empty,
                string.Empty);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patientDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            Log("Used PatientIdentifier Object: {0}", patientDemographic.GetType().Name);
            Log("-->FamilyName: {0}", patientDemographic.FamilyName);
            Log("-->GivenName: {0}", patientDemographic.GivenName);
            Log("-->DOB: {0}", patientDemographic.DateOfBirth);
            Log("-->Sex: {0}", patientDemographic.Sex);
            Log("-->DVA: {0}", patientDemographic.DvaNumber);
            Log("-->HospitalCode: {0}", patientDemographic.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", patientDemographic.HospitalCodeSystem);

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, "Individual details provided");
            LogAssert.AreEqual("Patient: No family name provided.", registrationResponse.HipsResponse.HipsErrorMessage, "Incorrect validation");
        }

        /// <summary>
        /// Test No:         84
        /// Objective/Input:
        ///
        /// Verify when the Identity of the individual has not been verified using a valid evidence of
        /// Identity by the healthcare provider, the request is not transmitted.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that the identity for the individual MUST be verified.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_084()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 17");

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            //change the behaviours of the request to remove the Evidence Of Identity - this will be passed by WCF and set the enum to 0
            AssistedRegistrationProxy proxy = new AssistedRegistrationProxy(string.Format("{0}_AssistedRegistrationService", ProxyHelper.EnvironmentUnderTest));
            InspectorBehavior newBehavior = new InspectorBehavior(XmlAction.Remove, xmlMatch: "<identityType>IdentityVerificationMethod1</identityType>");
            proxy.Endpoint.Behaviors.Add(newBehavior);

            PcehrRegistrationResponse registrationResponse = proxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.AreEqual(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse.Status, DialogueResource.HipsServiceUnverifiedIhiRegisterPatient);
            LogAssert.AreEqual("The Evidence Of Identity Type must be provided", registrationResponse.HipsResponse.HipsErrorMessage, "Incorrect validation");
        }


        /// <summary>
        /// Test No:         85
        /// Objective/Input:
        ///
        /// Perform a RegisterPCEHR operation with valid input elements to register
        /// a dependant. Ensure the dependant IHI Number is used, when dependant
        /// demographics are not provided in the request.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS)
        /// 
        /// ************************************************************************************************************
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_SetVisitorHospital")]
        public void PCEHR_NOC_085()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 27", preloadPatient: false);

            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 26", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.none;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetDemographic,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that the IHI number was included in the request
            XmlDocument xmlRequest;
            XmlNamespaceManager nsmgr;

            GetAuditRequest("Register PCEHR", out xmlRequest, out nsmgr);

            XmlNode ihiNumberNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.AreEqual(patient.IhiInformation.Ihi, ihiNumberNode.InnerText, "IHINumber element value");
        }

        /// <summary>
        /// Test No:         86
        /// Objective/Input:
        ///
        /// To show the ability of the client system to use the Medicare
        /// card Number and dependant demographics when dependant IHI number
        /// is not provided in the request.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS)
        /// 
        /// ************************************************************************************************************
        /// MUST CHANAGE THE "BypassHIService" configuration item to True to pass this test.
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_086()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 03", preloadPatient: false);

            // Create a demographic which contains no IHI
            // However HIPS will get an IHI from the HI service to send to the PCEHR
            Demographic demographics = patient.TargetDemographic;

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 11");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(demographics,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            XmlDocument xmlRequest;
            XmlNamespaceManager nsmgr;

            GetAuditRequest("Register PCEHR", out xmlRequest, out nsmgr);

            // Ensure the IHI header value is empty
            XmlNode ihiNumberNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            //LogAssert.AreEqual(string.Empty, ihiNumberNode.InnerText, "IHINumber element value");
            LogAssert.IsTrue(ihiNumberNode == null, "No IHI number recorded", "IHI number found");

            // Ensure the patient demographics were included
            XmlNode repFamilyNameNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:name/c:familyName", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.FamilyName, repFamilyNameNode.InnerText, "Family Name element value");
            XmlNode repGivenNameNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:name/c:givenName", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.GivenName, repGivenNameNode.InnerText, "Given Name element value");
            XmlNode repSexNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:sex", nsmgr);
            LogAssert.AreEqual(AuditHelper.GetSex(patient.TargetDemographic.Sex), repSexNode.InnerText, "Sex element value");
            XmlNode repDateOfBirthNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:dateOfBirth", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.DateOfBirth, DateTime.Parse(repDateOfBirthNode.InnerText), "Date of Birth element value");

            // Ensure the Medicare Number and IRN were included
            XmlNode repMedicareNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:medicareCardNumber", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.MedicareNumber, repMedicareNode.InnerText, "Medicare Card Number element value");
            XmlNode repMedicareIrnNode = xmlRequest.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:individual/p:demographics/c:medicareIRN", nsmgr);
            LogAssert.AreEqual(patient.TargetDemographic.MedicareIrn, repMedicareIrnNode.InnerText, "Medicare IRN element value");
        }

        /// <summary>
        /// Test No:         87
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending
        /// request to the PCEHR system when Representative declaration is
        /// not provided for dependant PCEHR registration, the request is not
        /// transmitted.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that representative
        /// declaration must be provided.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_087()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 03");

            // set the representativeDeclaration to false
            bool representativeDeclaration = false;

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 11");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetPatientIdentifier,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                representativeDeclaration,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         88
        /// Objective/Input:
        ///
        /// Verify that the client system can handle a Functional Error
        /// (PCEHR_ERROR_9009- child PCEHR already exists) when the PCEHR for
        /// the dependant already exists.
        ///
        /// Expected Result:
        ///
        /// The request is successfully transmitted to the PCEHR System and
        /// functional error PCEHR_ERROR_9009 is returned. Client system is able
        /// to handle the error and inform the user that the dependant PCEHR already exists.
        ///
        /// ************************************************************************************************************
        /// MUST CHANAGE THE "BypassHIService" configuration item to True to pass this test.
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_088()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 32", preloadPatient: false);

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 31", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetDemographic,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("PCEHR_ERROR_9009", registrationResponse.HipsResponse.ResponseCode, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         89
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending request to
        /// PCEHR system when the dependant PCEHR being registered is greater than 18
        /// years old, the request is not transmitted.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that a dependent cannot be
        /// older than 18 years of age.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_089()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 02");

            Demographic patientDemographic = patient.TargetDemographic;

            // Change the date of birth so the patient is greater than 18
            patientDemographic.DateOfBirth = new DateTime(1992, 1, 1);

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 10");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patientDemographic,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Dependant cannot be older than 18 years.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         90
        /// Objective/Input:
        ///
        /// Verify that the client system can handle a Functional Error
        /// (PCEHR_ERROR_9011- Parent and Child must be on the same Medicare Card)
        /// when the representative-dependant PCEHR being registered are not on the
        /// same Medicare card.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System. Client CIS or
        /// CSP system must inform the user that representative and dependant must be
        /// on the same Medicare card.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_090A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 02");

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 11");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetDemographic,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("The dependant and the representative must be on the same Medicare card.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         90
        /// Objective/Input:
        ///
        /// Verify that the client system can handle a Functional Error
        /// (PCEHR_ERROR_9011- Parent and Child must be on the same Medicare Card)
        /// when the representative-dependant PCEHR being registered are not on the
        /// same Medicare card.
        /// 
        /// ONLY IHI are used - Medicare Card Number IS Provided for the Child Demographics
        /// 
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System. Client CIS or
        /// CSP system must inform the user that representative and dependant must be
        /// on the same Medicare card.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_090B()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 02");

            ValidatedIhi patientValidatedIhi = new ValidatedIhi(patient.TestPatientValidIhi,
               IhiStatus.Active,
               IhiRecordStatus.Verified,
               DateTime.Now,
               patient.TargetDemographic.FamilyName,
               patient.TargetDemographic.GivenName,
               patient.TargetDemographic.DateOfBirth,
               patient.TargetDemographic.Sex,
               "RNH",
               "pasFacCd");

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 11");

            //duplicate the demographic
            Demographic childDemographic = patientRep.TargetDemographic;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patientValidatedIhi,
                childDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            Log("ValidatedIHI Provided for the Parent Demographics");
            Log("Medicare Card Number IS Provided for the Child Demographics");

            Log("Used Parent PatientIdentifier Object: {0}", patientValidatedIhi.GetType().Name);
            Log("-->IHI: {0}", patientValidatedIhi.Ihi);
            Log("-->IhiStatus: {0}", patientValidatedIhi.IhiStatus);
            Log("-->IhiRecordStatus: {0}", patientValidatedIhi.IhiRecordStatus);
            Log("-->IhiLastValidated: {0}", patientValidatedIhi.IhiLastValidated);
            Log("-->FamilyName: {0}", patientValidatedIhi.FamilyName);
            Log("-->GivenName: {0}", patientValidatedIhi.GivenName);
            Log("-->DOB: {0}", patientValidatedIhi.DateOfBirth);
            Log("-->Sex: {0}", patientValidatedIhi.Sex);
            Log("-->HospitalCode: {0}", patientValidatedIhi.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", patientValidatedIhi.HospitalCodeSystem);

            Log("Used Child PatientIdentifier Object: {0}", childDemographic.GetType().Name);
            Log("-->IHI: {0}", "");  //this is not part of the Demographic Object
            Log("-->FamilyName: {0}", childDemographic.FamilyName);
            Log("-->GivenName: {0}", childDemographic.GivenName);
            Log("-->Medicare Card: {0}", childDemographic.MedicareNumber);
            Log("-->DOB: {0}", childDemographic.DateOfBirth);
            Log("-->Sex: {0}", childDemographic.Sex);
            Log("-->HospitalCode: {0}", childDemographic.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", childDemographic.HospitalCodeSystem);

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("The dependant and the representative must be on the same Medicare card.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         90
        /// Objective/Input:
        ///
        /// Verify that the client system can handle a Functional Error
        /// (PCEHR_ERROR_9011- Parent and Child must be on the same Medicare Card)
        /// when the representative-dependant PCEHR being registered are not on the
        /// same Medicare card.
        /// 
        /// ONLY IHI are used - Medicare Card Number NOT Provided for the Child Demographics
        /// 
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System. Client CIS or
        /// CSP system must inform the user that representative and dependant must be
        /// on the same Medicare card.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_090C()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 02");

            ValidatedIhi patientValidatedIhi = new ValidatedIhi(patient.TestPatientValidIhi,
               IhiStatus.Active,
               IhiRecordStatus.Verified,
               DateTime.Now,
               patient.TargetDemographic.FamilyName,
               patient.TargetDemographic.GivenName,
               patient.TargetDemographic.DateOfBirth,
               patient.TargetDemographic.Sex,
               "RNH",
               "pasFacCd");

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 11");

            //duplicate the demographic and remove the Medicare Number
            Demographic childDemographic = patientRep.TargetDemographic;
            childDemographic.MedicareNumber = null;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patientValidatedIhi,
                childDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            Log("ValidatedIHI Provided for the Parent Demographics");
            Log("Medicare Card Number IS NOT Provided for the Child Demographics");
            Log("Used Parent PatientIdentifier Object: {0}", patientValidatedIhi.GetType().Name);
            Log("-->IHI: {0}", patientValidatedIhi.Ihi);
            Log("-->IhiStatus: {0}", patientValidatedIhi.IhiStatus);
            Log("-->IhiRecordStatus: {0}", patientValidatedIhi.IhiRecordStatus);
            Log("-->IhiLastValidated: {0}", patientValidatedIhi.IhiLastValidated);
            Log("-->FamilyName: {0}", patientValidatedIhi.FamilyName);
            Log("-->GivenName: {0}", patientValidatedIhi.GivenName);
            Log("-->DOB: {0}", patientValidatedIhi.DateOfBirth);
            Log("-->Sex: {0}", patientValidatedIhi.Sex);
            Log("-->HospitalCode: {0}", patientValidatedIhi.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", patientValidatedIhi.HospitalCodeSystem);

            Log("Used Child PatientIdentifier Object: {0}", childDemographic.GetType().Name);
            Log("-->IHI: {0}", "");  //this is not part of the Demographic Object
            Log("-->FamilyName: {0}", childDemographic.FamilyName);
            Log("-->GivenName: {0}", childDemographic.GivenName);
            Log("-->Medicare Card: {0}", childDemographic.MedicareNumber);
            Log("-->DOB: {0}", childDemographic.DateOfBirth);
            Log("-->Sex: {0}", childDemographic.Sex);
            Log("-->HospitalCode: {0}", childDemographic.HospitalCode);
            Log("-->HospitalCodeSystem: {0}", childDemographic.HospitalCodeSystem);

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Representative: Must include a Medicare or DVA number.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }


        /// <summary>
        /// Test No:         91
        /// Objective/Input:
        ///
        /// Verify that when an individual IHI Number is less than 14 years old in the
        /// PCEHRHeader, an error message appears and the request is not transmitted
        /// to the PCEHR system.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that the individual cannot be
        /// less than 14 years old.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_091()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 05");

            Demographic patientDemographics = patient.TargetDemographic;

            // Set patients age to less than 14
            patientDemographics.DateOfBirth = new DateTime(2006, 1, 1);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patientDemographics,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("An individual cannot be less than 14 years old.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         92
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending request to
        /// PCEHR system when there is less than 14-year age gap between representative
        /// and dependant for dependant PCEHR registration, the request is not transmitted.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System. Client CIS or CSP
        /// system must inform the user that there MUST be 14-year age gap between
        /// representative and dependent.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_092()
        {
            // Need to find patient with matching data
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 03");

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 11");

            Demographic repDemographics = patientRep.TargetDemographic;

            // Set age to within 14 years of dependant
            repDemographics.DateOfBirth = patient.TargetDemographic.DateOfBirth.AddYears(-10);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetDemographic,
                repDemographics,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("There cannot be less than a 14 year age gap between the dependant and the representative.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         93
        /// Objective/Input:
        ///
        /// Verify when the terms and conditions are not asserted and accepted by the
        /// healthcare provider, the request is not transmitted.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that an terms and conditions
        /// must be accepted.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_093()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 06");

            bool termsAndConditions = false;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                termsAndConditions,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("The latest terms and conditions have not been accepted.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         94
        /// Objective/Input:
        ///
        /// To show the ability of the client system to handle a standard error (PCEHR_ERROR_0505 - Invalid HPI-O) when an
        /// invalid HPI-O is used to perform a registerPCEHR operation. The client system is configured with an invalid HPI-O."
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System. Client system is able to handle and inform the user
        /// regarding the invalid HPI-O ( PCEHR_ERROR_0505 - Invalid HPI-O).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_094()
        {
            Log(@"Info: The HPI-O will be changed in the database from the valid HPI-O to an invalid HPI-O.");
            try
            {
                patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 07");
                HpiOHelper.ChangeHpiO(patient, this, makeValid: false);

                // Create blank correspondence channel
                RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
                channel.Channel = IvcCorrespondenceChannel.response;

                // Load BAD.exe file

                // Create document consent list
                List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() 
                {
                    new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                    new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                    new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                    new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                    new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                    new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                };

                PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                    EvidenceOfIdentityType.IdentityVerificationMethod1,
                    IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                    documentConsentList,
                    channel,
                    true,
                    patient.GetTestUser());

                LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
                if (patient.TargetHospital.PcehrCsp)
                {
                    LogAssert.AreEqual("notAuthorised", registrationResponse.HipsResponse.ResponseCode, "PCEHR Response Code");
                    LogAssert.AreEqual("PCEHR_ERROR_0004 - Authorisation denied", registrationResponse.HipsResponse.ResponseCodeDescription, "PCEHR Response Description");
                }
                else
                {
                    LogAssert.AreEqual("badParam", registrationResponse.HipsResponse.ResponseCode, "PCEHR Response Code");
                    LogAssert.AreEqual("PCEHR_ERROR_0505 - Invalid HPI-O", registrationResponse.HipsResponse.ResponseCodeDescription, "PCEHR Response Description");
                }
            }
            finally
            {
                HpiOHelper.ChangeHpiO(patient, this, makeValid: true);
            }
        }

        /// <summary>
        /// Test No:         95
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending a request
        /// to the PCEHR system, when the IVC Correspondence channel has not been selected.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR system.
        /// Client system should indicate to the user when the IVC Communication channel is
        /// not selected and that it MUST be supplied.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_095()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 06");

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("IVC Correspondence Channel has not been specified.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         96
        /// Objective/Input:
        ///
        /// Verify that the client system can handle a Functional Error
        /// (PCEHR_ERROR_9008- Individual PCEHR already exists) when the PCEHR for the
        /// individual already exists.
        ///
        /// Expected Result:
        ///
        /// The request is successfully transmitted to the PCEHR System and functional
        /// error PCEHR_ERROR_9008 is returned. Client system is able to handle the error
        /// and inform the user that the PCEHR already exists.
        /// 
        /// ************************************************************************************************************
        /// MUST CHANAGE THE "BypassHIService" configuration item to True to pass this test.
        /// MUST CHANGE "TestHospitalId" configuration item to the Hospital ID of the Visitor Hospital
        /// ************************************************************************************************************
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration_Bypass_Ihi_Retrieve")]
        public void PCEHR_NOC_096()
        {
            // Need to find patient with matching data
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 31", preloadPatient: false);

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("PCEHR_ERROR_9008", registrationResponse.HipsResponse.ResponseCode, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         97
        /// Objective/Input:
        ///
        /// Ensure the client system includes PCEHR Common Header Elements and Transmission
        /// Signature containing a signed attestation of elements contained within the SOAP
        /// message on all SOAP Requests.
        ///     - For this test case, the client system uses valid HPI-O certificates provided
        ///         in the test data package. Service Invoker and Service Provider SHALL
        ///         establish a secure communication channel when interacting with each other.
        ///     - Implementation confirms to web services base profile and TLS security profile.
        ///     - Unique message ID  is used in all transactions.
        ///     - IHI number is specified in all transactions, is valid, matches the individual
        ///         who owns the PCEHR and is numerical.
        ///     - ID is specified in all transactions, has no leading or trailing spaces. If
        ///         ID is HPII, ensure it is valid and numerical.
        ///     - If role and/or alternate organisation is supplied, no leading or trailing
        ///         spaces used
        ///     - User name, vendor, product name, production version, platform, organisation
        ///         ID, organisation name,  has no leading or trailing space, it is not blank.
        ///     - Organisation name is set to the name of the organisation.
        ///     - Product name, product version, product type and product platform are set to
        ///         the appropriate values as per client system.
        ///
        /// Expected Result:
        ///
        /// The requests are transmitted successfully to the PCEHR System.
        /// Response is PCEHR_SUCCESS (SUCCESS)
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_097()
        {
            // Need to find patient with matching data
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 06");

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         98
        /// Objective/Input:
        ///
        ///"Verify that the client system can handle a functional error (PCEHR_ERROR_9018 - Indigenous status has not been specified) 
        ///when the indigenous status has not been specified.
        ///
        /// Expected Result:
        ///
        /// The request is successfully transmitted to the PCEHR System and functional error PCEHR_ERROR_9018 is returned. 
        /// Client system is able to handle the error and inform the user that indigenous status has not been specified.
        /// 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_098()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 17");

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            //change the behaviours of the request to remove the Indigenous Status - this will be passed by WCF and set the enum to 0
            AssistedRegistrationProxy proxy = new AssistedRegistrationProxy(string.Format("{0}_AssistedRegistrationService", ProxyHelper.EnvironmentUnderTest));
            InspectorBehavior newBehavior = new InspectorBehavior(XmlAction.Remove, xmlMatch: "<indigenousStatus>NeitherAboriginalAndTorresStraitIslander</indigenousStatus>");
            proxy.Endpoint.Behaviors.Add(newBehavior);

            PcehrRegistrationResponse registrationResponse = proxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.AreEqual(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse.Status, DialogueResource.HipsServiceUnverifiedIhiRegisterPatient);
            LogAssert.AreEqual("The Indigenous Status must be provided", registrationResponse.HipsResponse.HipsErrorMessage, "Incorrect validation");
        }


        /// <summary>
        /// Test No:         99
        /// Objective/Input:
        ///
        /// Verify that the client system can handle a functional error
        /// (PCEHR_ERROR_0524 - Attachment exceeds maximum supported) when the
        /// attached consent form exceeds the supported size requirements.
        ///
        /// Expected Result:
        ///
        /// The request is successfully transmitted to the PCEHR System and
        /// functional error PCEHR_ERROR_0524 is returned. Client system is
        /// able to handle the error and inform the user the attachment exceeds
        /// maximum supported size.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_099()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 07");

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Load a large file that exceeds the file limit
            // Add the consent form
            Attachment consentForm = new Attachment();
            FileInfo file = new FileInfo(CcaPatient.TestDocumentPath + @"UC.CIS.501.20208h.pdf");
            consentForm.FileName = file.Name;
            consentForm.Contents = File.ReadAllBytes(file.FullName);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser(),
                consentForm);

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         100
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending a request to the PCEHR system, 
        /// when request with an attached signed consent form is not XOP/MTOM content type. 
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR system.
        /// 
        /// ***************************************************************************************************
        /// 
        /// This has been set as a positive test to show that the upload uses an MTOM request.
        /// 
        /// ***************************************************************************************************
        /// </summary>
        /// 
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_100_Positive()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 09");

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Load a large file that exceeds the file limit
            // Add the consent form
            Attachment consentForm = new Attachment();
            FileInfo file = new FileInfo(CcaPatient.TestDocumentPath + @"UC.CIS.501.20208h.png");
            consentForm.FileName = file.Name;
            consentForm.Contents = File.ReadAllBytes(file.FullName);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser(),
                consentForm);

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
        }


        /// <summary>
        /// Test No:         101
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending
        /// a request to the PCEHR system, when an unsupported MIME type is attached.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR system. Client system
        /// should inform the user that the mime type is not supported by the system.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_101()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 07");

            // Create blank correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Load invalid file (mpeg)
            // Add the consent form
            Attachment consentForm = new Attachment();
            FileInfo file = new FileInfo(CcaPatient.TestDocumentPath + @"UC.CIS.501.20208h.mpeg");
            consentForm.FileName = file.Name;
            consentForm.Contents = File.ReadAllBytes(file.FullName);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser(),
                consentForm);

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         102
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending a
        /// request to the PCEHR system, when the mobile IVC Correspondence
        /// channel is selected and a mobile number is not provided.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR system. Client
        /// system should inform the user that mobile number MUST be supplied.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_102()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 07");

            // Set the response channel to SMS but do not enter a phone
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.sms;
            channel.phoneNumber = string.Empty;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Mobile phone number is required for IVC SMS correspondence.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         103
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending
        /// a request to the PCEHR system, when the email IVC Correspondence
        /// channel is selected and an email address is not provided
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR system. Client
        /// system should inform the user that the email address MUST be supplied.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_103()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 07");

            // Set the response channel to email but do not enter an email address
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.email;
            channel.email = string.Empty;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Email address is required for IVC email correspondence.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         104
        /// Objective/Input:
        ///
        /// Perform a RegisterPCEHR operation for an Individual with valid input
        /// elements and select IVC Correspondence channel as "Response".
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully. Response is PCEHR_SUCCESS (SUCCESS)
        /// and the client system is able to inform the user of the IVC details
        /// received in the response.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_104()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 08");

            // Set the response channel to email but do not enter an email address
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
        }

        /// <summary>
        /// Test No:         105
        /// Objective/Input:
        ///
        /// To show the ability of the client system to prevent from sending a
        /// request to the PCEHR system, when request for dependant registration
        /// is requested with representative demographics that contains DVA file number.
        ///
        /// Expected Result:
        ///
        /// The request should not be transmitted to the PCEHR system. Client system
        /// should inform the user that representatives Medicare card number is required
        /// for dependant registration.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC_Assisted_Registration")]
        public void PCEHR_NOC_105()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 03");

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 11");

            Demographic repDemographic = patientRep.TargetDemographic;

            // Remove the Medicare number and Irn
            repDemographic.MedicareNumber = string.Empty;
            repDemographic.MedicareIrn = string.Empty;

            // Add a DVA number
            repDemographic.DvaNumber = "NX842654";

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() {
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetDemographic,
                repDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("The representatives Medicare number must be provided.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);
        }

        #region Private Methods

        private void GetAuditRequest(string requestName, out XmlDocument xmlRequest, out XmlNamespaceManager nsmgr)
        {
            PcehrAudit pcehrAudit;

            // Check that HIPS audited the request
            if (requestName != null && requestName != string.Empty)
            {
                pcehrAudit = patient.GetLastPcehrAudit(requestName);
            }
            else
            {
                pcehrAudit = patient.GetLastPcehrAudit();
            }

            // Confirm that the payload is well formed and complies with the gainPCEHRAccess Request XML Schema
            xmlRequest = new XmlDocument();
            using (StringReader sr = new StringReader(pcehrAudit.Request))
            {
                xmlRequest.Load(sr);
            }
            nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RegisterPCEHR/2.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
        }


        internal class MessageInspector : IClientMessageInspector
        {
            private XmlAction action;
            private string xmlMatch;
            private string xmlText;

            public MessageInspector(XmlAction action, string xmlMatch = null, string xmlText = null)
            {
                this.action = action;
                this.xmlMatch = xmlMatch;
                this.xmlText = xmlText;
            }

            public object BeforeSendRequest(ref System.ServiceModel.Channels.Message request, System.ServiceModel.IClientChannel channel)
            {
                string bigXml = ConvertMessageToString(request);

                switch (action)
                {
                    case XmlAction.Add:
                        //not yet implemented - use XPATH here
                        break;
                    case XmlAction.Remove:
                        //remove the xml item 
                        bigXml = Regex.Replace(bigXml, xmlMatch, "");
                        break;
                    case XmlAction.Replace:
                        //remove the xml item and replace
                        bigXml = Regex.Replace(bigXml, xmlMatch, xmlText);
                        break;
                    default:
                        break;
                }


                byte[] xmlArray = Encoding.UTF8.GetBytes(bigXml);
                XmlDictionaryReaderQuotas xdq = new XmlDictionaryReaderQuotas();

                var reader = XmlDictionaryReader.CreateTextReader(xmlArray, xdq);
                request = Message.CreateMessage(reader, int.MaxValue, request.Version);

                //msgBuffer.Close();

                return null;
            }
            public void AfterReceiveReply(ref System.ServiceModel.Channels.Message reply, object correlationState)
            {

            }
        }

        /// <summary>
        /// Convert the message to a string.
        /// </summary>
        /// <param name="msg">Message to convert.</param>
        /// <returns>Message as a string.</returns>
        private static string ConvertMessageToString(System.ServiceModel.Channels.Message msg)
        {
            var ms = new MemoryStream();
            var xw = XmlTextWriter.Create(ms, new XmlWriterSettings()
            {
                Indent = false,
                OmitXmlDeclaration = true
            });
            msg.WriteMessage(xw);
            xw.Close();
            ms.Position = 0;

            var sr = new StreamReader(ms);
            var convertedString = sr.ReadToEnd();

            sr.Close();
            ms.Close();

            return convertedString;
        }

        /// <summary>
        /// Implementation of a behaviour that instantiates a MessageInspector.
        /// </summary>
        internal class InspectorBehavior : IEndpointBehavior
        {
            private XmlAction action;
            private string xmlMatch;
            private string xmlText;

            public InspectorBehavior(XmlAction action, string xmlMatch = null, string xmlText = null)
            {
                this.action = action;
                this.xmlMatch = xmlMatch;
                this.xmlText = xmlText;
            }

            public void AddBindingParameters(ServiceEndpoint endpoint, System.ServiceModel.Channels.BindingParameterCollection bindingParameters)
            {
            }

            public void ApplyClientBehavior(ServiceEndpoint endpoint, System.ServiceModel.Dispatcher.ClientRuntime clientRuntime)
            {
                clientRuntime.MessageInspectors.Add(new MessageInspector(action, xmlMatch, xmlText));
            }

            public void ApplyDispatchBehavior(ServiceEndpoint endpoint, System.ServiceModel.Dispatcher.EndpointDispatcher endpointDispatcher)
            {
            }

            public void Validate(ServiceEndpoint endpoint)
            {
            }
        }

        public enum XmlAction
        {
            Remove = 1,
            Replace = 2,
            Add = 3
        }

        #endregion Private Methods
    }
}