﻿using System;
using System.Text;
using HIPS.Client.Proxy;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.ModelInterface.ConsentManagement;

namespace HIPS.Web.Data.Hips.ConsentManagement
{
    /// <summary>
    /// Implements a service for managing PCEHR consent. Includes the
    /// withdrawal/reinstatement of consent to upload documents for an
    /// episode, and the disclosure/removal of disclosure of the existence
    /// of a PCEHR.
    /// </summary>
    public class ConsentManagementService : IConsentManagementService
    {
        /// <summary>
        /// The client proxy.
        /// </summary>
        private ConsentProxy clientProxy = new ConsentProxy("ConsentEndPoint");

        /// <summary>
        /// Records a patient's withdrawal or reinstatement of consent to upload documents for a particular episode.
        /// </summary>
        /// <param name="consentWithdrawn">Boolean value indicating whether to record withdrawal (true) or reinstatement (false) of consent.</param>
        /// <param name="notes">Notes entered by the user.</param>
        /// <param name="patientIdentifier">The MRN that identifies the patient at the hospital.</param>
        /// <param name="admissionDate">The admission date that identifies which episode the consent should be recorded against.</param>
        /// <param name="user">The identity of the user who is operating the application.</param>
        /// <returns>A service response that indicates whether the operation was successful or not.</returns>
        public ServiceResponse<bool> RecordConsent(bool consentWithdrawn, string notes, Mrn patientIdentifier, DateTime admissionDate, CommonSchemas.UserDetails user)
        {
            var response = this.clientProxy.RecordConsent(consentWithdrawn, patientIdentifier, admissionDate, user, Encoding.UTF8.GetBytes(notes));

            bool isSuccessful = false;
            ResponseMessageList messages = new ResponseMessageList();

            if (response != null)
            {
                isSuccessful = response.Status == HipsResponseIndicator.OK;
                messages.AddRange(response.ToMessageListExpectOkStatus());
            }
            else
            {
                messages.Add("Unable to get response", MessageLevel.Error);
            }

            return new ServiceResponse<bool>(consentWithdrawn, isSuccessful, messages);
        }

        /// <summary>
        /// Records a patient's disclosure or removal of disclosure of the existence of a PCEHR.
        /// </summary>
        /// <param name="pcehrDisclosed">Boolean value indicating whether to record disclosure (true) or removal of disclosure (false).</param>
        /// <param name="notes">Notes entered by the user.</param>
        /// <param name="patientIdentifier">The MRN that identifies the patient at the hospital.</param>
        /// <param name="userDetails">The identity of the user who is operating the application.</param>
        /// <returns>A service response that indicates whether the operation was successful or not.</returns>
        public ServiceResponse<bool> RecordDisclosure(bool pcehrDisclosed, string notes, Mrn patientIdentifier, CommonSchemas.UserDetails userDetails)
        {
            var response = this.clientProxy.RecordDisclosure(pcehrDisclosed, patientIdentifier, userDetails, Encoding.UTF8.GetBytes(notes));

            bool isSuccessful = false;
            ResponseMessageList messages = new ResponseMessageList();

            if (response != null)
            {
                isSuccessful = response.Status == HipsResponseIndicator.OK;
                messages.AddRange(response.ToMessageListExpectOkStatus());
            }
            else
            {
                messages.Add("Unable to get response", MessageLevel.Error);
            }

            return new ServiceResponse<bool>(pcehrDisclosed, isSuccessful, messages);
        }

        /// <summary>
        /// Safely disposes the client proxy.
        /// </summary>
        public void Dispose()
        {
            if (this.clientProxy != null)
            {
                this.clientProxy.DisposeSafely();
                this.clientProxy = null;
            }
        }
    }
}