﻿using System;
using System.Collections.Generic;

using HIPS.Client.Proxy;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Data.Hips.Mapping;
using HIPS.Web.ModelInterface.DocumentManagement;

namespace HIPS.Web.Data.Hips.DocumentManagement
{
    /// <summary>
    /// Implements a repository for interacting with documents that have been
    /// uploaded to the PCEHR system by this instance of HIPS.
    /// </summary>
    public class UploadedDocumentRepository : RepositoryBase<PCEHRProxy>, IUploadedDocumentRepository
    {
        /// <summary>
        /// Cda Document Details cache key.
        /// </summary>
        private const string CDA_DOCUMENT_DETAILS_KEY = "CdaDocumentDetails";

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="UploadedDocumentRepository"/> class.
        /// </summary>
        public UploadedDocumentRepository()
            : this(new HIPS.Web.Components.Cache.NoCachingCacheProvider(), null)
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="UploadedDocumentRepository"/> class.
        /// </summary>
        /// <param name="cacheProvider">Cache provider to be employed by the repository.</param>
        /// <param name="cacheKeyPrefix">Key prefix to be employed for caching.</param>
        public UploadedDocumentRepository(ICacheProvider cacheProvider, string cacheKeyPrefix = "")
            : base(cacheProvider, cacheKeyPrefix)
        {
            // Initialise client proxy ready for use.
            this.ClientProxy = new PCEHRProxy("PCEHREndPoint");
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// List the uploaded documents for a certain patient at a certain hospital.
        /// If setId is specified, lists all versions of the one document.
        /// If setId is not specified, lists the current version of all documents.
        /// </summary>
        /// <param name="user">Current user details.</param>
        /// <param name="setId">Identifier of the document set whose documents are to be listed.</param>
        /// <param name="patientIdentifier">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and list of documents.</returns>
        public ServiceResponse<LocalClinicalDocumentListResponse<PatientIdentifierBase>> List(UserDetails user, string setId, Mrn patientIdentifier)
        {
            var result = this.ClientProxy.ListUploadedDocuments(user, setId, patientIdentifier);
            ResponseMessageList message = new ResponseMessageList();
            bool isSuccessful = false;
            if (result != null)
            {
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;
                message.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                message.Add("Unable to List the documents", MessageLevel.Error);
            }
            return new ServiceResponse<LocalClinicalDocumentListResponse<PatientIdentifierBase>>(result, isSuccessful, message);
        }

        /// <summary>
        /// Retrieve an uploaded document from the cache if available, otherwise from the HIPS system.
        /// </summary>
        /// <param name="user">Current user details.</param>
        /// <param name="setId">Identifier of the document set.</param>
        /// <param name="documentId">Identifier of the document instance.</param>
        /// <param name="patientIdentifer">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and document contents.</returns>
        public ServiceResponse<LocalClinicalDocumentResponse<PatientIdentifierBase>> Get(UserDetails user, string setId, string documentId, Mrn patientIdentifer)
        {
            return CacheProvider.GetOrSetIf(
                this.GetMethodInvocationFullCacheKey(
                 System.Reflection.MethodInfo.GetCurrentMethod().Name,
                    new object[] { setId, documentId }),
                () => this._Get(user, setId, documentId, patientIdentifer),
                response => response.IsSuccessful);
        }

        /// <summary>
        /// Enqueue a request for HIPS to remove an uploaded document from the PCEHR system.
        /// This marks the document set as 'removed' which makes it invisible to the patient
        /// and to other healthcare provider organisations. The status will be reset to
        /// 'uploaded' if a new version of the document is uploaded.
        /// </summary>
        /// <param name="patientIdentifier">Identifier of the patient and hospital.</param>
        /// <param name="admissionDate">Admission date and time used to identify the episode against which the document was uploaded.</param>
        /// <param name="setId">Identifier of the document set.</param>
        /// <param name="reason">Reason for removal: Withdrawn or IncorrectIdentity.</param>
        /// <param name="user">Current user details.</param>
        /// <param name="auditInformation">Reference notes entered by the user.</param>
        /// <returns>Response containing status information.</returns>
        public ServiceResponse<HipsResponse> Remove(Mrn patientIdentifier, DateTime admissionDate, string setId, RemovalReason reason, UserDetails user, byte[] auditInformation)
        {
            var result = this.ClientProxy.Remove(patientIdentifier, admissionDate, setId, reason, user, auditInformation);
            ResponseMessageList message = new ResponseMessageList();
            bool isSuccessful = false;
            if (result != null)
            {
                isSuccessful = result.Status == HipsResponseIndicator.OK;
                message.AddRange(result.ToMessageListExpectOkStatus());
            }
            else
            {
                message.Add("Unable to remove the document", MessageLevel.Error);
            }
            return new ServiceResponse<HipsResponse>(result, isSuccessful, message);
        }

        /// <summary>
        /// Uploads a Level 3A discharge summary to the PCEHR National Repository,
        /// optionally superseding a previously-uploaded document. This method
        /// takes in the XML document and any attachments and does the packaging
        /// itself.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="cdaDocument">The CDA root XML document.</param>
        /// <param name="attachments">The attachments.</param>
        /// <param name="admissionDate">The admission date.</param>
        /// <param name="documentFormatCode">The document format code.</param>
        /// <param name="userContext">User requesting the operation.</param>
        /// <returns>Response containing status information.</returns>
        public ServiceResponse<HipsResponse> UploadOrSupersedeDocument(
            Model.PatientIdentifier.Mrn patientIdentifier,
            byte[] cdaDocument,
            List<Model.Cda.CdaAttachment> attachments,
            DateTime admissionDate,
            string documentFormatCode,
            UserDetails userContext)
        {
            var result = this.ClientProxy.UploadOrSupersedeDocument(
              cdaDocument,
              ObjectMapper.Map<Mrn>(patientIdentifier),
              userContext,
              ObjectMapper.Map<List<Attachment>>(attachments).ToArray(),
              admissionDate,
              documentFormatCode);

            ResponseMessageList message = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                isSuccessful = result.Status == HipsResponseIndicator.OK;
                message.AddRange(result.ToMessageListExpectOkStatus());
            }
            else
            {
                message.Add("Unable to upload/supersede the document", MessageLevel.Error);
            }

            return new ServiceResponse<HipsResponse>(result, isSuccessful, message);
        }

        /// <summary>
        /// Cache a CdaDocumentDetails.
        /// </summary>
        /// <param name="setId">Identifier for the document.</param>
        /// <param name="cda">CdaDocumentDetails to be cached.</param>
        public void SetCdaDocumentDetails(string setId, HIPS.Web.Model.Cda.CdaDocumentDetails cda)
        {
            this.CacheProvider.Set<HIPS.Web.Model.Cda.CdaDocumentDetails>(this.GetMethodInvocationFullCacheKey(CDA_DOCUMENT_DETAILS_KEY, setId), cda);
        }

        /// <summary>
        /// Gets CdaDocumentDetails from cache.
        /// </summary>
        /// <param name="setId">Identifier for the document.</param>
        /// <returns>A CdaDocumentDetails.</returns>
        public HIPS.Web.Model.Cda.CdaDocumentDetails GetCdaDocumentDetails(string setId)
        {
            return this.CacheProvider.Get<HIPS.Web.Model.Cda.CdaDocumentDetails>(this.GetMethodInvocationFullCacheKey(CDA_DOCUMENT_DETAILS_KEY, setId));
        }

        #endregion Methods

        #region Private

        /// <summary>
        /// Retrieves an uploaded document from the HIPS system.
        /// </summary>
        /// <param name="user">Current user details.</param>
        /// <param name="setId">Identifier of the document set.</param>
        /// <param name="documentId">Identifier of the document instance.</param>
        /// <param name="patientIdentifer">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and document contents.</returns>
        private ServiceResponse<LocalClinicalDocumentResponse<PatientIdentifierBase>> _Get(UserDetails user, string setId, string documentId, Mrn patientIdentifer)
        {
            var result = this.ClientProxy.GetLocalUploadedDocument(user, setId, documentId, patientIdentifer);
            ResponseMessageList message = new ResponseMessageList();
            bool isSuccessful = false;
            if (result != null)
            {
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;
                message.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                message.Add("Unable to remove the document", MessageLevel.Error);
            }
            return new ServiceResponse<LocalClinicalDocumentResponse<PatientIdentifierBase>>(result, isSuccessful, message);
        }

        #endregion Private
    }
}