﻿using HIPS.Client.Proxy;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PatientSchemas;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;

namespace HIPS.Web.Data.Hips
{
    /// <summary>
    /// Implements a repository for interacting with patients.
    /// </summary>
    public class PatientRepository : RepositoryBase<PatientProxy>, HIPS.Web.ModelInterface.Common.IPatientRepository
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="PatientRepository"/> class.
        /// </summary>
        public PatientRepository()
            : this(0, new HIPS.Web.Components.Cache.NoCachingCacheProvider(), null)
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="PatientRepository"/> class.
        /// </summary>
        /// <param name="daysDischarged">Integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".</param>
        /// <param name="cacheProvider">Cache provider to be employed by the repository.</param>
        /// <param name="cacheKeyPrefix">Key prefix to be employed for caching.</param>
        public PatientRepository(int daysDischarged, ICacheProvider cacheProvider, string cacheKeyPrefix = "")
            : base(cacheProvider, cacheKeyPrefix)
        {
            // Initialise client proxy ready for use.
            this.ClientProxy = new PatientProxy("PatientEndPoint");
            this.DaysDischarged = daysDischarged;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets an integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".
        /// </summary>
        public int DaysDischarged { get; private set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Retrieves a list of patients who are currently admitted to (and have not been discharged from) a specified hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientListResponse containing the results of the operation.</returns>
        public ServiceResponse<PatientSchemas.PatientListResponse> ListPatientsCurrentlyInHospital(string hospitalCodeSystem, string hospitalCode, bool? withPcehr, CommonSchemas.UserDetails operatingUser)
        {
            return CacheProvider.GetOrSetIf(
                this.GetMethodInvocationFullCacheKey(
                    System.Reflection.MethodInfo.GetCurrentMethod().Name,
                    new object[] { hospitalCodeSystem, hospitalCode, withPcehr, this.DaysDischarged }),
                () => this._ListPatientsCurrentlyInHospital(hospitalCodeSystem, hospitalCode, withPcehr, this.DaysDischarged, operatingUser),
                response => response.IsSuccessful);
        }

        /// <summary>
        /// Retrieves a list of patients who are currently admitted or recently discharged from a specified hospital.
        /// </summary>
        /// <param name="user">User requesting the operation.</param>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withIhi">Boolean value indicating whether to list patients with (true) or without (false) a verified active IHI. Null indicates all patients should be listed regardless of whether they have an IHI.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="excludeMedicareExclusions">Boolean value indicating whether to exclude (true) or include (false) patients with a Medicare Care Number that exists in the HIPS MedicareExclusions table.</param>
        /// <returns>ServiceResponse containing AdmittedPatientListResponse containing the results of the operation.</returns>
        public ServiceResponse<PatientSchemas.AdmittedPatientListResponse> ListAdmittedPatients(UserDetails user, string hospitalCodeSystem, string hospitalCode, bool? withIhi, bool? withPcehr, bool excludeMedicareExclusions)
        {
            return CacheProvider.GetOrSetIf(
                this.GetMethodInvocationFullCacheKey(
                    System.Reflection.MethodInfo.GetCurrentMethod().Name,
                    new object[] { hospitalCodeSystem, hospitalCode, withIhi, withPcehr, excludeMedicareExclusions, this.DaysDischarged }),
                () => this._ListAdmittedPatients(hospitalCodeSystem, hospitalCode, withIhi, withPcehr, excludeMedicareExclusions, this.DaysDischarged, user),
                response => response.IsSuccessful);
        }

        /// <summary>
        /// Retrieves the details associated with the disclosure of the existence of a PCEHR by a specified patient.
        /// </summary>
        /// <param name="user">User requesting the operation.</param>
        /// <param name="patientIdentifer">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and the details associated with the disclosure.</returns>
        public ServiceResponse<PatientDisclosureDetailsResponse> GetPatientDisclosureDetails(UserDetails user, Mrn patientIdentifer)
        {
            var result = this.ClientProxy.GetPatientDisclosureDetails(user, patientIdentifer);
            ResponseMessageList message = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;
                message.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                message.Add("Unable to retrieve patient disclosure details", MessageLevel.Error);
            }

            return new ServiceResponse<PatientDisclosureDetailsResponse>(result, isSuccessful, message);
        }

        /// <summary>
        /// Retrieves the details associated with the disclosure of the existence of a PCEHR by a specified patient.
        /// </summary>
        /// <param name="user">User requesting the operation.</param>
        /// <param name="patientIdentifer">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and the details associated with the disclosure.</returns>
        public ServiceResponse<PatientDisclosureDetailsResponse> GetPatientDisclosureDetails(UserDetails user, RegisteredEnterprisePatient patientIdentifer)
        {
            var result = this.ClientProxy.GetPatientDisclosureDetails(user, patientIdentifer);
            ResponseMessageList message = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;
                message.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                message.Add("Unable to retrieve patient disclosure details", MessageLevel.Error);
            }

            return new ServiceResponse<PatientDisclosureDetailsResponse>(result, isSuccessful, message);
        }

        #region Private

        /// <summary>
        /// Retrieves a list of patients who are currently admitted to (and have not been discharged from) a specified hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="daysDischarged">Integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientListResponse containing the results of the operation.</returns>
        private ServiceResponse<PatientSchemas.PatientListResponse> _ListPatientsCurrentlyInHospital(string hospitalCodeSystem, string hospitalCode, bool? withPcehr, int daysDischarged, CommonSchemas.UserDetails operatingUser)
        {
            var result = this.ClientProxy.ListPatientsInHospital(operatingUser, hospitalCodeSystem, hospitalCode, withPcehr, daysDischarged);

            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to retrieve patients for the specified hospital.", MessageLevel.Error);
            }

            return new ServiceResponse<PatientSchemas.PatientListResponse>(result, isSuccessful, messages);
        }       

        /// <summary>
        /// Retrieves a list of patients who are currently admitted or recently discharged from a specified hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withIhi">Boolean value indicating whether to list patients with (true) or without (false) a verified active IHI. Null indicates all patients should be listed regardless of whether they have an IHI.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="excludeMedicareExclusions">Boolean value indicating whether to exclude (true) or include (false) patients with a Medicare Care Number that exists in the HIPS MedicareExclusions table.</param>
        /// <param name="daysDischarged">Integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".</param>
        /// <param name="user">User requesting the operation.</param>
        /// <returns>ServiceResponse containing AdmittedPatientListResponse containing the results of the operation.</returns>
        private ServiceResponse<PatientSchemas.AdmittedPatientListResponse> _ListAdmittedPatients(string hospitalCodeSystem, string hospitalCode, bool? withIhi, bool? withPcehr, bool excludeMedicareExclusions, int daysDischarged, UserDetails user)
        {
            var result = this.ClientProxy.ListAdmittedPatients(hospitalCodeSystem, hospitalCode, withIhi, withPcehr, excludeMedicareExclusions, daysDischarged, user);

            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to retrieve patients for the specified hospital.", MessageLevel.Error);
            }

            return new ServiceResponse<PatientSchemas.AdmittedPatientListResponse>(result, isSuccessful, messages);
        }

        #endregion Private

        #endregion Methods
        
    }
}