﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;

using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.CommonSchemas;
using HIPS.PatientBusinessLogic;
using HIPS.PcehrDataStore.Schemas;
using HIPS.ServiceContracts.Common.Fault;
using HIPS.ServiceContracts.Patient;
using HIPS.ServiceContracts.Patient.Message;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// Implementation of web service operations on patients in the HIPS database.
    /// </summary>
    public partial class PatientService : IPatientServiceV2
    {
        #region Methods

        /// <summary>
        /// Lists the episodes for a specific patient in a defined hospital
        /// </summary>
        /// <param name="request">Request containing a patient identifier for retrieving patient episodes.</param>
        /// <returns>Response containing the a list of the admitted or discharged episodes for the patient with in the specified number of days</returns>
        public ListPatientEpisodesInHospitalResponse ListPatientEpisodesInHospital(ListPatientEpisodesInHospitalRequest request)
        {
            this.ValidateRequest(request);

            try
            {
                // Map DTO to Common Model
                UserDetails user = ObjectMapper.Map<UserDetails>(request.User);
                var patientIdentifier = ObjectMapper.Map<HIPS.CommonSchemas.PatientIdentifier.PatientIdentifierBase>(request.PatientIdentifier);

                // Invoke Business Service
                List<PatientInHospitalEpisode> data;
                
                var logic = new PatientListLogic();                
                var hipsResponse = logic.ListPatientEpisodesInHospital(user, patientIdentifier, request.DaysDischarged, out data, request.IncludeDocuments, request.DocumentTypeCode);
                var messages = new List<string>() { hipsResponse.HipsErrorMessage, hipsResponse.ResponseCodeDescription, hipsResponse.ResponseCodeDetails };
                
                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<List<HIPS.ServiceContracts.Common.DTO.PatientEpisode>>(data);                
                var serviceMessages = ObjectMapper.Map<List<ServiceContracts.Common.DTO.Message>>(messages.Where(message => !string.IsNullOrEmpty(message)));
                var serviceStatus = (hipsResponse.Status == HipsResponseIndicator.OK ? ServiceContracts.Common.ResponseStatus.OK : ServiceContracts.Common.ResponseStatus.Warning);

                return new ListPatientEpisodesInHospitalResponse() { Status = serviceStatus, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        #endregion Methods
    }
}