﻿using System;
using System.IO;
using System.Runtime.Serialization;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Ihi;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrQueueLogic;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.PCEHR.RemoveDocument;

using Enums = HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public class DocumentRemovalBeforeQueue
    {
        #region Private Properties

        private DateTime AdmissionDate { get; set; }

        private PatientAccess PatientAccess { get; set; }

        private PatientIdentifierBase PatientIdentifier
        {
            get
            {
                return QueuedRemoveOperation.PatientIdentifier;
            }
            set
            {
                QueuedRemoveOperation.PatientIdentifier = value;
            }
        }

        private QueuedRemoveOperation QueuedRemoveOperation { get; set; }

        #endregion Private Properties

        #region Public Properties

        /// <summary>
        /// Gets the hospital that has been loaded in the LoadPatientEpisode method.
        /// </summary>
        public Hospital Hospital { get { return QueuedRemoveOperation.Hospital; } }

        /// <summary>
        /// Gets the patient master that has been loaded in the LoadPatientEpisode method.
        /// </summary>
        public PatientMaster PatientMaster { get { return QueuedRemoveOperation.PatientMaster; } }

        #endregion Public Properties

        /// <summary>
        /// Creates an instance of DocumentRemovalBeforeQueue.
        /// </summary>
        /// <param name="user">The person responsible for the removal action</param>
        public DocumentRemovalBeforeQueue(PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user, string documentSetId)
        {
            this.QueuedRemoveOperation = new QueuedRemoveOperation(user);
            this.QueuedRemoveOperation.SourceSystemSetId = documentSetId;
            this.PatientIdentifier = patientIdentifier;
            this.AdmissionDate = admissionDate;
            this.PatientAccess = new PatientAccess(user);
        }

        /// <summary>
        /// Loads the hospital, hospital patient, patient master and episode.
        /// Also checks the validity of the IHI. If there are outstanding alerts then access is blocked.
        /// If the IHI was last validated outside the configured period then HIPS will attempt to validate the IHI.
        /// If the IHI is still invalid after the attempt to validate, then access is blocked.
        /// </summary>
        /// <returns>Information about success or failure</returns>
        public HipsResponse LoadPatientEpisode()
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            PatientMaster patientMaster;
            Hospital hospital;
            HospitalPatient hospitalPatient;
            response = PatientAccess.GetHospital(PatientIdentifier, out hospital);
            if (response.Status != HipsResponseIndicator.OK)
            {
                return response;
            }

            // It is alright if the the patient has an invalid IHI because we should try to validate it.
            response = PatientAccess.GetPatient(PatientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (response.Status != HipsResponseIndicator.OK && response.Status != HipsResponseIndicator.InvalidIhi)
            {
                return response;
            }

            // If the IHI was last validated outside the configured period then HIPS will attempt to validate it.
            IhiSearchResponse ihiResponse = new PatientIhiValidation().GetValidatedIhi(PatientIdentifier, hospital, QueuedRemoveOperation.User, patientMaster);
            if (ihiResponse.HipsResponse.Status == HipsResponseIndicator.HiServiceError)
            {
                // If the IHI cannot be validated because the HI service is unavailable, HIPS will
                // allow the remove operation to go onto the queue.

                EventLogger.WriteLog(ResponseStrings.InfoRemoveQueuedWithStaleIhi, new Exception(ihiResponse.HipsResponse.HipsErrorMessage), QueuedRemoveOperation.User, LogMessage.HIPS_MESSAGE_085);
                response.Status = HipsResponseIndicator.OK;
            }
            else
            {
                if (ihiResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    return ihiResponse.HipsResponse;
                }

                // But now if the IHI is still invalid then we must prevent access to the PCEHR.
                PatientAccess.ValidateLocalIhiInformation(patientMaster, response);
                if (response.Status != HipsResponseIndicator.OK && patientMaster.IhiStatusId != (int)IhiStatus.ServiceUnavailable)
                {
                    return response;
                }
            }

            //A Specific Episode that has not been cancelled must be found in this case
            Episode episode = PatientAccess.GetEpisodeWithoutCancelled(PatientIdentifier, AdmissionDate, hospitalPatient, QueuedRemoveOperation.SourceSystemSetId);
            if (episode == null)
            {
                response.Status = HipsResponseIndicator.InvalidEpisode;
                return response;
            }
            QueuedRemoveOperation.Hospital = hospital;
            QueuedRemoveOperation.HospitalPatient = hospitalPatient;
            QueuedRemoveOperation.PatientMaster = patientMaster;
            QueuedRemoveOperation.Episode = episode;
            return response;
        }

        /// <summary>
        /// Adds an operation on the queue to remove a clinical document from the PCEHR repository.
        /// </summary>
        /// <param name="documentSetId">The identifier for the document (shared by all versions of the same document)</param>
        /// <param name="reason">The code indicating the reason for removal (withdrawn by provider, or incorrect identity)</param>
        /// <param name="auditInformation">Additional audit information to store</param>
        /// <returns>Information about success or failure</returns>
        public HipsResponse Remove(Enums.RemovalReason reason, byte[] auditInformation)
        {
            removeDocumentReasonForRemoval nehtaReason;
            switch (reason)
            {
                case Enums.RemovalReason.IncorrectIdentity:
                    nehtaReason = removeDocumentReasonForRemoval.IncorrectIdentity;
                    break;

                case Enums.RemovalReason.Withdrawn:
                    nehtaReason = removeDocumentReasonForRemoval.Withdrawn;
                    break;

                default:
                    return new HipsResponse(HipsResponseIndicator.SystemError);
            }

            this.QueuedRemoveOperation.RemovalReason = nehtaReason;
            this.QueuedRemoveOperation.AuditInformation = auditInformation;

            return PlaceOperationOnQueue();
        }

        /// <summary>
        /// Populates the pending item, saves the operation in the PCEHR Data Store and adds it to the MSMQ queue.
        /// </summary>
        /// <returns>Response explaining any failure to store in database or MSMQ.</returns>
        private HipsResponse PlaceOperationOnQueue()
        {
            // Ensure that the other methods have been executed successfully first.
            this.QueuedRemoveOperation.PendingItem = new PcehrMessageQueue();
            this.QueuedRemoveOperation.PendingItem.QueueOperationId = (int)QueueOperation.Remove;
            this.QueuedRemoveOperation.PendingItem.QueueStatusId = (int)QueueStatus.Pending;
            this.QueuedRemoveOperation.PendingItem.EpisodeId = QueuedRemoveOperation.Episode.EpisodeId.Value;
            this.QueuedRemoveOperation.PendingItem.SourceSystemSetId = QueuedRemoveOperation.SourceSystemSetId;
            this.QueuedRemoveOperation.PendingItem.SourceSystemDocumentId = null; //document id is not know for the remove operation

            using (MemoryStream stream = new MemoryStream())
            {
                new DataContractSerializer(typeof(QueuedUploadOperation)).WriteObject(stream, this.QueuedRemoveOperation);
                this.QueuedRemoveOperation.PendingItem.SerialisedObject = stream.GetBuffer();
            }

            DocumentQueueTransactionHandler handler = new DocumentQueueTransactionHandler(this.QueuedRemoveOperation.User);
            return handler.SaveRemoveOperationAndPlaceOnQueue(this.QueuedRemoveOperation);
        }
    }
}