﻿using System;
using System.Runtime.Serialization;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.ConsentSchemas
{
    /// <summary>
    /// Represents a consent, details and audit reason for an episode
    /// The audit reason will only be displayed if the episode has a current withdrawn consent
    /// </summary>
    [KnownType(typeof(EpisodeConsentResponse))]
    [KnownType(typeof(PatientEpisodeDetailedResponse))]
    [Serializable]
    [DataContract]
    public class PatientEpisodeDetailedResponse
    {
        #region Private Members

        private SexEnumerator currentSex = SexEnumerator.NotStatedOrInadequatelyDescribed;
        private HipsResponse hipsResponse = new HipsResponse(HipsResponseIndicator.SystemError, null);

        #endregion Private Members

        #region Properties

        /// <summary>
        /// The admission date of the patient.
        /// </summary>
        [DataMember]
        public DateTime AdmissionDate { get; set; }

        /// <summary>
        /// The admission reason of the patient
        /// </summary>
        [DataMember]
        public string AdmissionReason { get; set; }

        /// <summary>
        /// The current Bed of the patient.
        /// </summary>
        [DataMember]
        public string Bed { get; set; }

        /// <summary>
        /// Whether or not the consent has been withdrawn
        /// </summary>
        [DataMember]
        public bool? ConsentWithdrawn { get; set; }

        /// <summary>
        /// The patient's current sex.
        /// </summary>
        [DataMember]
        public SexEnumerator CurrentSex
        {
            get { return currentSex; }
            set { currentSex = value; }
        }

        /// <summary>
        /// The patient's current sex identifier.
        /// </summary>
        public int CurrentSexId { get; set; }

        /// <summary>
        /// The patient's Date of Birth.
        /// </summary>
        [DataMember]
        public DateTime DateOfBirth { get; set; }

        /// <summary>
        /// The discharge date of the patient.
        /// </summary>
        [DataMember]
        public DateTime? DischargeDate { get; set; }

        /// <summary>
        /// The patient's family name.
        /// </summary>
        [DataMember]
        public string FamilyName { get; set; }

        /// <summary>
        /// The patient's given names.
        /// </summary>
        [DataMember]
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets the HipsResponse.
        /// Common Response Object.
        /// </summary>
        /// <value>
        /// The hipsResponse
        /// </value>
        [DataMember]
        public HipsResponse HipsResponse
        {
            get { return hipsResponse; }
            set { hipsResponse = value; }
        }

        /// <summary>
        /// The code of the hospital where the patient is or was admitted.
        /// </summary>
        [DataMember]
        public string HospitalCode { get; set; }

        /// <summary>
        /// The internal primary key of the hospital where the patient is or was admitted.
        /// </summary>
        public int HospitalId { get; set; }

        /// <summary>
        /// The name of the hospital where the patient is or was admitted.
        /// </summary>
        [DataMember]
        public string HospitalName { get; set; }

        /// <summary>
        /// The patient's MRN at the hospital where the patient was admitted.
        /// </summary>
        [DataMember]
        public string Mrn { get; set; }

        /// <summary>
        /// The family name of the patient's responsible provider.
        /// </summary>
        [DataMember]
        public string ResponsibleProviderFamilyName { get; set; }

        /// <summary>
        /// The given names of the patient's responsible provider.
        /// </summary>
        [DataMember]
        public string ResponsibleProviderGivenNames { get; set; }

        /// <summary>
        /// The suffix of the patient's responsible provider (such as 'Jr' or 'Sr')
        /// </summary>
        [DataMember]
        public string ResponsibleProviderSuffix { get; set; }

        /// <summary>
        /// The title of the patient's responsible provider (such as 'Dr', 'Ms' or 'Mr').
        /// </summary>
        [DataMember]
        public string ResponsibleProviderTitle { get; set; }

        /// <summary>
        /// The current Room of the patient.
        /// </summary>
        [DataMember]
        public string Room { get; set; }

        /// <summary>
        /// The source system episode id or visit number, specific to the hospital
        /// </summary>
        [DataMember]
        public string SourceSystemEpisodeId { get; set; }

        /// <summary>
        /// The patient's name suffixes (like "Jr")
        /// </summary>
        [DataMember]
        public string Suffix { get; set; }

        /// <summary>
        /// The patient's titles (like "Dr", "Ms" or "Mr")
        /// </summary>
        [DataMember]
        public string Title { get; set; }

        /// <summary>
        /// The current Ward of the patient.
        /// </summary>
        [DataMember]
        public string Ward { get; set; }

        /// <summary>
        /// The Withdrawal audit information
        /// </summary>
        [DataMember]
        public byte[] WithdrawalAuditInformation { get; set; }

        #endregion Properties
    }
}