﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrDataStore.Schemas.Schemas;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the Address table.
    /// </summary>
    public class AddressDl : BaseDl
    {
        /// <summary>
        /// Gets the specified name id.
        /// </summary>
        /// <param name="contactId">The name id.</param>
        /// <returns></returns>
        public Address Get(int addressId)
        {
            Address result = new Address();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.AddressGet"))
                {
                    command.Parameters.Add(new SqlParameter("@AddressId", addressId));
                    PopulateBusinessObject<Address>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageAddressGet, ex, User, LogMessage.HIPS_MESSAGE_004);
            }
            return result;
        }

        /// <summary>
        /// Gets all by patient.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="transaction">An existing transaction (optional)</param>
        /// <returns>List of addresses</returns>
        public List<Address> GetAllByPatient(int patientMasterId, SqlTransaction transaction = null)
        {
            List<Address> results = new List<Address>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.AddressByPatientList", transaction))
                {
                    command.Parameters.Add(new SqlParameter("@PatientMasterId", patientMasterId));
                    results = GetPopulatedBusinessList<Address>(command.ExecuteReader());
                    if (transaction == null)
                    {
                        command.Connection.Close();
                    }
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageAddress, ex, User, LogMessage.HIPS_MESSAGE_005);
            }
            return results;
        }

        /// <summary>
        /// Updates the specified episode.
        /// </summary>
        /// <param name="episode">The episode.</param>
        /// <returns></returns>
        public bool Update(Address episode, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.AddressUpdate", transaction))
            {
                return base.Update<Address>(episode, command);
            }
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Insert(Address item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.AddressInsert", transaction))
            {
                return base.Insert<Address>(item, command);
            }
        }

        /// <summary>
        /// Deletes the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Delete(Address item, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.AddressDelete", transaction))
            {
                return base.Delete<Address>(item, command);
            }
        }

        /// <summary>
        /// Updates the specified names for a patient.
        /// </summary>
        /// <param name="names">The names.</param>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <returns></returns>
        public bool Update(List<Address> addresses, int patientMasterId, SqlTransaction transaction = null)
        {
            if (addresses.Count == 0)
            {
                return true;
            }
            bool updateResult = true;
            List<Address> originalAddresses = GetAllByPatient(patientMasterId, transaction);
            List<Address> newAddresses = addresses.FindAll(address => address.AddressId == null);
#if PUMA_CLIENT
            List<Address> updatedAddresses = addresses.FindAll(address => address.AddressId != null && address.IsDirty);
#else
            List<Address> updatedAddresses = addresses.FindAll(address => address.AddressId != null);
#endif

            IEnumerable<int?> originalIds = from result in originalAddresses
                                            select result.AddressId;
            IEnumerable<int?> currentIds = from result in addresses
                                           select result.AddressId;
            IEnumerable<int?> newIds = from result in newAddresses
                                       select result.AddressId;
            IEnumerable<int?> deletedIds = originalIds.Except(currentIds);

            updateResult = DeleteAddresses(transaction, updateResult, originalAddresses, deletedIds);

            if (updateResult)
            {
                updateResult = InsertAddresses(patientMasterId, transaction, updateResult, newAddresses);
            }
            if (updateResult)
            {
                updateResult = UpdateAddresses(patientMasterId, transaction, updateResult, updatedAddresses);
            }

            return updateResult;
        }

        /// <summary>
        /// Inserts the new addresses.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="newAddresses">The new addresses.</param>
        /// <returns></returns>
        private bool InsertAddresses(int patientMasterId, SqlTransaction transaction, bool updateResult, List<Address> newAddresses)
        {
            foreach (Address address in newAddresses)
            {
                updateResult = Insert(address, transaction);
                if (updateResult)
                {
                    updateResult = InsertPatientLink(patientMasterId, address.AddressId.Value, transaction);
                }
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Deletes the removed addresses.
        /// </summary>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="originalAddresses">The original addresses.</param>
        /// <param name="deletedIds">The deleted ids.</param>
        /// <returns></returns>
        private bool DeleteAddresses(SqlTransaction transaction, bool updateResult, List<Address> originalAddresses, IEnumerable<int?> deletedIds)
        {
            foreach (int deletedId in deletedIds)
            {
                updateResult = Delete(originalAddresses.Find(result => result.AddressId == deletedId), transaction);
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Updates the existing changed addresses.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <param name="updateResult">if set to <c>true</c> [update result].</param>
        /// <param name="modifiedAddresses">The modified addresses.</param>
        /// <returns></returns>
        private bool UpdateAddresses(int patientMasterId, SqlTransaction transaction, bool updateResult, List<Address> modifiedAddresses)
        {
            foreach (Address address in modifiedAddresses)
            {
                updateResult = Update(address, transaction);
                if (!updateResult)
                {
                    break;
                }
            }
            return updateResult;
        }

        /// <summary>
        /// Inserts the patient link record.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="contactId">The name id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        public bool InsertPatientLink(int patientMasterId, int addressId, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.PatientMasterAddressInsert", transaction))
            {
                PatientMasterAddress item = new PatientMasterAddress(patientMasterId, addressId);
                return base.Insert<PatientMasterAddress>(item, command);
            }
        }

        /// <summary>
        /// Deletes the patient link.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="contactId">The name id.</param>
        /// <param name="connection">The connection.</param>
        /// <param name="transaction">The transaction.</param>
        /// <returns></returns>
        public bool DeletePatientLink(int patientMasterId, int addressId, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.PatientMasterAddressDelete", transaction))
            {
                PatientMasterAddress item = new PatientMasterAddress(patientMasterId, addressId);
                return base.Delete<PatientMasterAddress>(item, command);
            }
        }
    }
}