﻿using System;
using System.Runtime.Serialization;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.Schemas.Schemas
{
    /// <summary>
    /// This class represents an Australian or International Address, belonging either to a Hospital or a PatientMaster.
    /// </summary>
    [KnownType(typeof(Address))]
    [Serializable]
    [DataContract]
    public class Address : BaseSchema
    {
        #region Private Members

        private int? hospitalId;
        private string addressLine1;
        private string addressLine2;
        private string placeName;
        private int? australianStateId;
        private string australianStateName;
        private string australianStateCode;
        private string internationalStateCode;
        private string postCode;
        private int countryId;
        private string countryName;
        private int addressTypeId;
        private string addressTypeDescription;

        #endregion Private Members

        #region Constructors

        /// <summary>
        /// Initializes a new instance of the <see cref="Address"/> class.
        /// </summary>

        public Address()
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="Address"/> class.
        /// </summary>
        /// <param name="addressLine1">The address line1.</param>
        /// <param name="addressLine2">The address line2.</param>
        /// <param name="placeName">Name of the place.</param>
        /// <param name="australianStateId">The australian state id.</param>
        /// <param name="internationalStateCode">The international state code.</param>
        /// <param name="postCode">The post code.</param>
        /// <param name="addressTypeId">The address type id.</param>
        /// <param name="countryId">The country id.</param>
        public Address(string addressLine1, string addressLine2, string placeName, int australianStateId, string internationalStateCode, string postCode, int addressTypeId, int countryId)
        {
            AddressLine1 = addressLine1;
            AddressLine2 = addressLine2;
            PlaceName = placeName;
            AustralianStateId = australianStateId;
            InternationalStateCode = internationalStateCode;
            PostCode = postCode;
            AddressTypeId = addressTypeId;
            CountryId = countryId;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets the address id.
        /// </summary>
        /// <value>
        /// The address id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.AutoGeneratedKey)]
        public int? AddressId
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }

        /// <summary>
        /// Gets or sets the hospital ID. This will only be valued if this address belongs to a hospital.
        /// </summary>
        /// <value>
        /// The hospital id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int? HospitalId
        {
            get
            {
                return hospitalId;
            }
            set
            {
                hospitalId = value;
            }
        }

        /// <summary>
        /// Gets or sets the First line of unstructured address
        /// </summary>
        /// <value>
        /// The address line 1.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string AddressLine1
        {
            get
            {
                return addressLine1;
            }
            set
            {
                addressLine1 = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => AddressLine1);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the Second line of unstructured address
        /// </summary>
        /// <value>
        /// The address line 2.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string AddressLine2
        {
            get
            {
                return addressLine2;
            }
            set
            {
                addressLine2 = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => AddressLine2);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the Suburb, Town or Locality
        /// </summary>
        /// <value>
        /// The name of the place.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string PlaceName
        {
            get
            {
                return placeName;
            }
            set
            {
                placeName = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => PlaceName);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the australian state id.
        /// </summary>
        /// <value>
        /// The australian state id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int? AustralianStateId
        {
            get
            {
                return australianStateId;
            }
            set
            {
                australianStateId = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => AustralianStateId);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the name of the australian state.
        /// </summary>
        /// <value>
        /// The name of the australian state.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string AustralianStateCode
        {
            get
            {
                return australianStateCode;
            }
            set
            {
                australianStateCode = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => AustralianStateCode);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the name of the australian state.
        /// </summary>
        /// <value>
        /// The name of the australian state.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string AustralianStateName
        {
            get
            {
                return australianStateName;
            }
            set
            {
                australianStateName = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => AustralianStateName);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the state or province for international addresses.
        /// </summary>
        /// <value>
        /// The international state code.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string InternationalStateCode
        {
            get
            {
                return internationalStateCode;
            }
            set
            {
                internationalStateCode = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => InternationalStateCode);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the post code (ZIP) for national and international addresses.
        /// </summary>
        /// <value>
        /// The post code.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string PostCode
        {
            get
            {
                return postCode;
            }
            set
            {
                postCode = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => PostCode);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the country id.
        /// </summary>
        /// <value>
        /// The country id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int CountryId
        {
            get
            {
                return countryId;
            }
            set
            {
                countryId = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => CountryId);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the name of the country.
        /// </summary>
        /// <value>
        /// The name of the country.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CountryName
        {
            get
            {
                return countryName;
            }
            set
            {
                countryName = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => CountryName);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the address type id.
        /// </summary>
        /// <value>
        /// The address type id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int AddressTypeId
        {
            get
            {
                return addressTypeId;
            }
            set
            {
                addressTypeId = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => AddressTypeId);
#endif
            }
        }

        /// <summary>
        /// Gets or sets the address type description.
        /// </summary>
        /// <value>
        /// The address type description.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string AddressTypeDescription
        {
            get
            {
                return addressTypeDescription;
            }
            set
            {
                addressTypeDescription = value;
#if PUMA_CLIENT
                RaisePropertyChanged(() => AddressTypeDescription);
#endif
            }
        }

        /// <summary>
        /// Gets the compare key, for comparing two Addresses to see if they are the same.
        /// </summary>
        public string CompareKey
        {
            get
            {
                string address1 = (AddressLine1 ?? string.Empty).Trim().ToUpper();
                string address2 = (AddressLine2 ?? string.Empty).Trim().ToUpper();
                string placeName = (PlaceName ?? string.Empty).Trim().ToUpper();
                string internationalStateCode = (InternationalStateCode ?? string.Empty).Trim().ToUpper();
                string postCode = (PostCode ?? string.Empty).Trim().ToUpper();
                return string.Format("{0}{1}{2}{3}{4}{5}{6}", address1, address2, placeName, AustralianStateId, internationalStateCode, postCode, CountryId, AddressTypeId).ToUpper();
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Creates a copy of this address.
        /// </summary>
        /// <returns>A copy of this address.</returns>
        internal Address CloneForNewPatientMaster()
        {
            Address clone = this.MemberwiseClone() as Address;
            clone.AddressId = null;
            return clone;
        }

        #endregion Methods
    }
}