﻿using System;
using System.Data.SqlClient;
using System.IO;
using System.Runtime.Serialization;
using System.Transactions;

using HIPS.Client.Proxy;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;

namespace HIPS.PcehrQueueLogic
{
    /// <summary>
    /// This class allows transactional saving of clinical documents, clinical document versions and PCEHR message queue items.
    /// </summary>
    public class DocumentQueueTransactionHandler : BaseDl
    {
        private ClinicalDocumentDl documentDataAccess;
        private PcehrMessageQueueDl queueDataAccess;
        private RemoveAuditDl removeAuditDataAccess;
        private ClinicalDocumentVersionDl versionDataAccess;

        #region Constructor

        public DocumentQueueTransactionHandler(UserDetails user)
        {
            documentDataAccess = new ClinicalDocumentDl(user);
            versionDataAccess = new ClinicalDocumentVersionDl(user);
            queueDataAccess = new PcehrMessageQueueDl(user);
            removeAuditDataAccess = new RemoveAuditDl(user);
        }

        #endregion Constructor

        #region Methods

        /// <summary>
        /// Loads the PcehrMessageQueue item from the database, restores the
        /// serialized object in the pending item (so that it's not lost on
        /// an update) and deserializes the QueuedUploadOperation to extract
        /// the CDA package and restore it into the QueuedUploadOperation.
        /// The Operation serialised from the queue is replaced with the object from the database, as the
        /// package can be too large for the MSMQ upper limit of 2MB.
        /// Additionally, if the DBA changes the Operation Status this will be picked up by this process and
        /// not simply extracted from the message in the MSMQ object - this is important as it allows a failing MSMQ message
        /// to be set to failed from the DB rather than continuously processing on the queue.
        /// </summary>
        /// <param name="operation">The queued upload operation that has come
        /// through MSMQ with the package removed.</param>
        /// <returns>Whether the package was successfully restored.</returns>
        public bool RestorePackageInQueuedUploadOperation(QueuedUploadOperation operation)
        {
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                try
                {
                    PcehrMessageQueue storedItem;
                    if (queueDataAccess.Get(operation.PendingItem.PcehrMessageQueueId.Value, out storedItem))
                    {
                        operation.PendingItem.QueueStatusId = storedItem.QueueStatusId;
                        operation.PendingItem.QueueStatusDescription = storedItem.QueueStatusDescription;
                        operation.PendingItem.SourceSystemSetId = storedItem.SourceSystemSetId;
                        operation.PendingItem.SourceSystemDocumentId = storedItem.SourceSystemDocumentId;
                        operation.PendingItem.SerialisedObject = storedItem.SerialisedObject;
                        operation.PendingItem.DateModified = storedItem.DateModified;
                        using (MemoryStream stream = new MemoryStream(storedItem.SerialisedObject))
                        {
                            QueuedUploadOperation deserializedOperation = new DataContractSerializer(typeof(QueuedUploadOperation)).ReadObject(stream) as QueuedUploadOperation;
                            if (deserializedOperation != null)
                            {
                                operation.Package = deserializedOperation.Package;
                                return true;
                            }
                        }
                    }
                }
                catch (Exception ex)
                {
                    operation.PendingItem.QueueStatusId = (int)QueueStatus.Failure;
                    operation.PendingItem.Details = string.Format(ResponseStrings.CouldNotReloadPackage, operation.PendingItem.PcehrMessageQueueId);
                    EventLogger.WriteLog(operation.PendingItem.Details, ex, User, LogMessage.HIPS_MESSAGE_073);
                    UpdateOperationForFailure(operation.PendingItem);
                }
            }
            return false;
        }

        /// <summary>
        /// Saves the document and either deletes or updates the pending item, after a successful remove from PCEHR.
        /// Whether to delete or update the PDS queue item is controlled by configuration.
        /// </summary>
        /// <param name="document">The document set (required)</param>
        /// <param name="pendingItem">The successful queued operation (required)</param>
        /// <exception cref="InvalidOperationException">If any of the database operations fail, to force the MSMQ to retry the operation.</exception>
        public void SaveAfterSuccessfulRemove(ClinicalDocument document, QueuedRemoveOperation operation)
        {
            PcehrMessageQueue pendingItem = operation.PendingItem;
            bool saved = false;
            SqlTransaction transaction = null;

            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                using (SqlCommand command = base.Command)
                {
                    try
                    {
                        transaction = command.Connection.BeginTransaction();
                        command.Transaction = transaction;
                        saved = documentDataAccess.Save(document, transaction);
                        if (saved)
                        {
                            if (Settings.Instance.DeleteQueuedItemOnSuccess)
                            {
                                saved = queueDataAccess.Delete(pendingItem, transaction);
                            }
                            else
                            {
                                saved = queueDataAccess.Update(pendingItem, transaction);
                            }
                        }
                        if (saved)
                        {
                            RemoveAudit audit = new RemoveAudit();
                            audit.AuditInformation = operation.AuditInformation;
                            audit.ClinicalDocumentId = document.ClinicalDocumentId.Value;
                            audit.RemovalReasonId = document.RemovalReasonId;
                            saved = removeAuditDataAccess.Insert(audit, transaction);
                        }
                        if (saved)
                        {
                            transaction.Commit();
                        }
                        else
                        {
                            transaction.Rollback();
                            throw new InvalidOperationException(LogStrings.ErrorSuccessfulRemoveSave);
                        }
                    }
                    catch (Exception ex)
                    {
                        if (transaction != null)
                        {
                            transaction.Rollback();
                        }
                        command.Connection.Close();
                        EventLogger.WriteLog(LogStrings.ErrorSuccessfulRemoveSave, ex, User, LogMessage.HIPS_MESSAGE_072);
                        throw new InvalidOperationException(LogStrings.ErrorSuccessfulRemoveSave, ex);
                    }
                }
            }
        }

        /// <summary>
        /// Saves the document, old version and new version and either deletes or updates the pending item, after a successful upload to PCEHR.
        /// Whether to delete or update the item is controlled by configuration.
        /// </summary>
        /// <param name="document">The document set (required)</param>
        /// <param name="oldCurrentVersion">The version that was superseded (optional)</param>
        /// <param name="newCurrentVersion">The version that was uploaded (required)</param>
        /// <param name="pendingItem">The successful queued operation (required)</param>
        /// <exception cref="InvalidOperationException">If any of the database operations fail, to force the MSMQ to retry the operation.</exception>
        public void SaveAfterSuccessfulUpload(ClinicalDocument document, ClinicalDocumentVersion oldCurrentVersion, ClinicalDocumentVersion newCurrentVersion, PcehrMessageQueue pendingItem)
        {
            bool saved = false;
            SqlTransaction transaction = null;

            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                using (SqlCommand command = base.Command)
                {
                    try
                    {
                        transaction = command.Connection.BeginTransaction();
                        command.Transaction = transaction;
                        saved = documentDataAccess.Save(document, transaction);
                        if (saved)
                        {
                            if (oldCurrentVersion != null)
                            {
                                oldCurrentVersion.SupersededDate = DateTime.Now;
                                saved = versionDataAccess.Save(oldCurrentVersion, transaction);
                            }
                        }
                        if (saved)
                        {
                            // For a new clinicalDocument, the ID has only just been allocated
                            newCurrentVersion.ClinicalDocumentId = document.ClinicalDocumentId.Value;
                            saved = versionDataAccess.Save(newCurrentVersion, transaction);
                        }
                        if (saved)
                        {
                            if (Settings.Instance.DeleteQueuedItemOnSuccess)
                            {
                                saved = queueDataAccess.Delete(pendingItem, transaction);
                            }
                            else
                            {
                                saved = queueDataAccess.Update(pendingItem, transaction);
                            }
                        }
                        if (saved)
                        {
                            transaction.Commit();
                        }
                        else
                        {
                            transaction.Rollback();
                            throw new InvalidOperationException(LogStrings.ErrorSuccessfulUploadSave);
                        }
                    }
                    catch (Exception ex)
                    {
                        if (transaction != null)
                        {
                            transaction.Rollback();
                        }
                        command.Connection.Close();
                        EventLogger.WriteLog(LogStrings.ErrorSuccessfulUploadSave, ex, User, LogMessage.HIPS_MESSAGE_071);
                        throw new InvalidOperationException(LogStrings.ErrorSuccessfulUploadSave, ex);
                    }
                }
            }
        }

        /// <summary>
        /// Saves the remove operation in the PCEHR Data Store and places the operation on the HIPS PCEHR MSMQ queue.
        /// If the MSMQ operation fails (e.g. queue full or faulted), the database transaction will be rolled back.
        /// </summary>
        /// <param name="documentUpload">The document upload operation.</param>
        public HipsResponse SaveRemoveOperationAndPlaceOnQueue(QueuedRemoveOperation queuedRemoveOperation)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            bool saved = false;
            SqlTransaction transaction = null;

            using (SqlCommand command = base.Command)
            {
                try
                {
                    transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;

                    saved = queueDataAccess.Insert(queuedRemoveOperation.PendingItem, transaction);

                    if (saved)
                    {
                        using (PcehrQueueProxy proxy = new PcehrQueueProxy("PcehrQueueEndPoint"))
                        {
                            try
                            {
                                proxy.SendRemoveRequest(queuedRemoveOperation);
                                saved = true;
                            }
                            catch (Exception ex)
                            {
                                proxy.Abort();
                                EventLogger.WriteLog(LogStrings.ErrorAddingRemoveOperationToMsmq, ex, queuedRemoveOperation.User, LogMessage.HIPS_MESSAGE_067);
                                response = new HipsResponse(HipsResponseIndicator.CouldNotAddToQueue, ResponseStrings.CouldNotAddRemoveToQueue);
                                saved = false;
                            }
                        }
                    }

                    if (saved)
                    {
                        transaction.Commit();
                    }
                    else
                    {
                        transaction.Rollback();
                    }
                }
                catch (Exception ex)
                {
                    if (transaction != null)
                    {
                        transaction.Rollback();
                    }
                    command.Connection.Close();
                    EventLogger.WriteLog(LogStrings.ErrorInsertingRemoveOperation, ex, User, LogMessage.HIPS_MESSAGE_068);
                    response = new HipsResponse(HipsResponseIndicator.CouldNotAddToQueue, ResponseStrings.CouldNotAddRemoveToQueue);
                }
            }

            return response;
        }

        /// <summary>
        /// Saves the operation in the PCEHR Data Store (with the SerialisedObject containing the CDA package),
        /// then removes the package and the serialised object from the operation before placing it on the HIPS
        /// PCEHR MSMQ queue (to ensure it is under the 4MB limit). If the MSMQ operation fails (e.g. queue full
        /// or faulted), the database transaction will be rolled back.
        /// </summary>
        /// <param name="documentUpload">The document upload operation.</param>
        public HipsResponse SaveUploadOperationAndPlaceOnQueue(QueuedUploadOperation queuedUploadOperation)
        {
            HipsResponse response = new HipsResponse(HipsResponseIndicator.OK);
            bool saved = false;
            SqlTransaction transaction = null;

            using (SqlCommand command = base.Command)
            {
                try
                {
                    transaction = command.Connection.BeginTransaction();
                    command.Transaction = transaction;

                    saved = queueDataAccess.Insert(queuedUploadOperation.PendingItem, transaction);

                    if (saved)
                    {
                        // Remove the package and serialised object from the item before it goes on the queue.
                        // This should ensure it doesn't exceed MSMQ's 4MB limit.
                        queuedUploadOperation.Package = null;
                        queuedUploadOperation.PendingItem.SerialisedObject = null;

                        using (PcehrQueueProxy proxy = new PcehrQueueProxy("PcehrQueueEndPoint"))
                        {
                            try
                            {
                                proxy.SendUploadRequest(queuedUploadOperation);
                                saved = true;
                            }
                            catch (Exception ex)
                            {
                                proxy.Abort();
                                EventLogger.WriteLog(LogStrings.ErrorAddingUploadOperationToMsmq, ex, queuedUploadOperation.User, LogMessage.HIPS_MESSAGE_069);
                                response = new HipsResponse(HipsResponseIndicator.CouldNotAddToQueue, ResponseStrings.CouldNotAddUploadToQueue);
                                saved = false;
                            }
                        }
                    }

                    if (saved)
                    {
                        transaction.Commit();
                    }
                    else
                    {
                        transaction.Rollback();
                    }
                }
                catch (Exception ex)
                {
                    if (transaction != null)
                    {
                        transaction.Rollback();
                    }
                    command.Connection.Close();
                    EventLogger.WriteLog(LogStrings.ErrorInsertingUploadOperation, ex, User, LogMessage.HIPS_MESSAGE_070);
                    response = new HipsResponse(HipsResponseIndicator.CouldNotAddToQueue, ResponseStrings.CouldNotAddUploadToQueue);
                }
            }

            return response;
        }

        /// <summary>
        /// Updates the record in the PCEHR Data Store relating to the current queue operation.
        /// </summary>
        /// <param name="item">The record in the PCEHR Data Store</param>
        /// <exception cref="InvalidOperationException">If the database update fails, to force the MSMQ to retry the operation.</exception>
        public void UpdateOperationForFailure(PcehrMessageQueue item)
        {
            bool result;
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                result = queueDataAccess.Update(item);
            }
            if (!result)
            {
                throw new InvalidOperationException(LogStrings.ErrorSavingFailedOperation);
            }
        }

        #endregion Methods
    }
}