﻿using System;
using System.Linq;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.PCEHR;

namespace HIPS.PcehrSchemas
{
    /// <summary>
    /// A set of static functions used to interrogate the NEHTA PCEHR enums.
    /// Specifically helping with the CodedValueAttribute extension attributes
    /// </summary>
    public class PcehrEnumHelper
    {
        /// <summary>
        /// Initialises a new instance of the <see cref="PcehrEnumHelper"/> class.
        /// </summary>
        private PcehrEnumHelper()
        {
        }

        /// <summary>
        /// Finds a member of the ClassCodes enumeration by matching the text in a particular field of the CodedValueAttribute attribute.
        /// </summary>
        /// <param name="attributeField">Which field of the attribute to match on.</param>
        /// <param name="text">The text to look for.</param>
        /// <returns>The ClassCodes enumeration member that matched.</returns>
        public static ClassCodes GetClassCode(CodedValueAttributeField attributeField, string text)
        {
            try
            {
                ClassCodes code;

                switch (attributeField)
                {
                    case CodedValueAttributeField.ConceptCode:
                        code = Enum.GetValues(typeof(ClassCodes)).Cast<ClassCodes>()
                                    .Where(v => v.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptCode) == text)
                                    .First();
                        break;

                    case CodedValueAttributeField.ConceptName:
                        code = Enum.GetValues(typeof(ClassCodes)).Cast<ClassCodes>()
                                    .Where(v => v.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptName) == text)
                                    .First();
                        break;

                    case CodedValueAttributeField.AlternateName:
                        code = Enum.GetValues(typeof(ClassCodes)).Cast<ClassCodes>()
                                    .Where(v => v.GetAttributeValue<CodedValueAttribute, string>(a => a.AlternateName) == text)
                                    .First();

                        break;

                    default:
                        throw new CodedValueAttribException();
                }

                return code;
            }
            catch (InvalidOperationException e)
            {
                //if an InvalidOperationException is thrown then return the exception object
                throw new Exception("ClassCode with the Attribute field '" + attributeField.ToString() + "' equalling '" + text + "' does not exist", e);
            }
            catch (CodedValueAttribException)
            {
                //if an CodedValueAttribException is thrown then return the exception object
                throw new Exception("CodedValueAttributeField '" + attributeField.ToString() + "' not available in the current ClassCodes context");
            }
        }

        /// <summary>
        /// Gets the value of a specified field in the CodedValueAttribute attribute on the given ClassCodes enumeration member.
        /// </summary>
        /// <param name="classCode">The enumeration member.</param>
        /// <param name="attributeField">Which attribute field to get the value of.</param>
        /// <returns>The value of the field.</returns>
        public static string GetClassCodeAttributeValue(ClassCodes classCode, CodedValueAttributeField attributeField)
        {
            string attributeValue = "";
            try
            {
                switch (attributeField)
                {
                    case CodedValueAttributeField.ConceptCode:
                        attributeValue = classCode.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptCode);
                        break;

                    case CodedValueAttributeField.ConceptName:
                        attributeValue = classCode.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptName);
                        break;

                    case CodedValueAttributeField.AlternateName:
                        attributeValue = classCode.GetAttributeValue<CodedValueAttribute, string>(a => a.AlternateName);
                        break;

                    case CodedValueAttributeField.CodingSystem:
                        attributeValue = classCode.GetAttributeValue<CodedValueAttribute, string>(a => a.CodingSystem);
                        break;

                    case CodedValueAttributeField.CodingSystemOID:
                        attributeValue = classCode.GetAttributeValue<CodedValueAttribute, string>(a => a.CodingSystemOID);
                        break;

                    default:
                        throw new CodedValueAttribException();
                }

                return attributeValue;
            }
            catch (InvalidOperationException)
            {
                //if an InvalidOperationException is thrown then return empty string
                return attributeValue;
            }
            catch (CodedValueAttribException)
            {
                //if an CodedValueAttribException is thrown then return the exception object
                throw new Exception("CodedValueAttributeField '" + attributeField.ToString() + "' not available in the current ClassCodes context");
            }
        }

        /// <summary>
        /// Finds a member of the HealthcareFacilityTypeCodes enumeration by matching
        /// the text in a particular field of the CodedValueAttribute attribute.
        /// </summary>
        /// <param name="attributeField">Which field of the attribute to match on.</param>
        /// <param name="text">The text to look for.</param>
        /// <returns>The HealthcareFacilityTypeCodes enumeration member that matched.</returns>
        public static HealthcareFacilityTypeCodes GetHealthcareFacilityTypeCode(CodedValueAttributeField attributeField, String text)
        {
            try
            {
                HealthcareFacilityTypeCodes code;

                switch (attributeField)
                {
                    case CodedValueAttributeField.ConceptCode:
                        code = Enum.GetValues(typeof(HealthcareFacilityTypeCodes)).Cast<HealthcareFacilityTypeCodes>()
                                    .Where(v => v.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptCode) == text)
                                    .First();
                        break;

                    case CodedValueAttributeField.ConceptName:
                        code = Enum.GetValues(typeof(HealthcareFacilityTypeCodes)).Cast<HealthcareFacilityTypeCodes>()
                                    .Where(v => v.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptName) == text)
                                    .First();
                        break;

                    default:
                        throw new CodedValueAttribException();
                }

                return code;
            }
            catch (InvalidOperationException e)
            {
                //if an InvalidOperationException is thrown then return the exception object
                throw new Exception("HealthcareFacilityTypeCodes with the Attribute field '" + attributeField.ToString() + "' equalling '" + text + "' does not exist", e);
            }
            catch (CodedValueAttribException)
            {
                //if an CodedValueAttribException is thrown then return the exception object
                throw new Exception("CodedValueAttributeField '" + attributeField.ToString() + "' not available in the current HealthcareFacilityTypeCodes context");
            }
        }

        /// <summary>
        /// Gets the value of a specified field in the CodedValueAttribute attribute
        /// on the given HealthcareFacilityTypeCodes enumeration member.
        /// </summary>
        /// <param name="healthcareFacilityTypeCode">The enumeration member.</param>
        /// <param name="attributeField">Which attribute field to get the value of.</param>
        /// <returns>The value of the field.</returns>
        public static String GetHealthcareFacilityTypeCodeAttributeValue(HealthcareFacilityTypeCodes healthcareFacilityTypeCode, CodedValueAttributeField attributeField)
        {
            String attributeValue = "";
            try
            {
                switch (attributeField)
                {
                    case CodedValueAttributeField.ConceptCode:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptCode);
                        break;

                    case CodedValueAttributeField.ConceptName:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptName);
                        break;

                    case CodedValueAttributeField.CodingSystem:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.CodingSystem);
                        break;

                    case CodedValueAttributeField.CodingSystemOID:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.CodingSystemOID);
                        break;

                    default:
                        throw new CodedValueAttribException();
                }

                return attributeValue;
            }
            catch (InvalidOperationException)
            {
                //if an InvalidOperationException is thrown then return empty string
                return attributeValue;
            }
            catch (CodedValueAttribException)
            {
                //if an CodedValueAttribException is thrown then return the exception object
                throw new Exception("CodedValueAttributeField '" + attributeField.ToString() + "' not available in the current HealthcareFacilityTypeCodes context");
            }
        }

        /// <summary>
        /// Finds a member of the PracticeSettingTypes enumeration by matching
        /// the text in a particular field of the CodedValueAttribute attribute.
        /// </summary>
        /// <param name="attributeField">Which field of the attribute to match on.</param>
        /// <param name="text">The text to look for.</param>
        /// <returns>The PracticeSettingTypes enumeration member that matched.</returns>
        public static PracticeSettingTypes GetPracticeSettingType(CodedValueAttributeField attributeField, String text)
        {
            try
            {
                PracticeSettingTypes code;

                switch (attributeField)
                {
                    case CodedValueAttributeField.ConceptCode:
                        code = Enum.GetValues(typeof(PracticeSettingTypes)).Cast<PracticeSettingTypes>()
                                    .Where(v => v.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptCode) == text)
                                    .First();
                        break;

                    case CodedValueAttributeField.ConceptName:
                        code = Enum.GetValues(typeof(PracticeSettingTypes)).Cast<PracticeSettingTypes>()
                                    .Where(v => v.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptName) == text)
                                    .First();
                        break;

                    default:
                        throw new CodedValueAttribException();
                }

                return code;
            }
            catch (InvalidOperationException e)
            {
                //if an InvalidOperationException is thrown then return the exception object
                throw new Exception("PracticeSettingTypes with the Attribute field '" + attributeField.ToString() + "' equalling '" + text + "' does not exist", e);
            }
            catch (CodedValueAttribException)
            {
                //if an CodedValueAttribException is thrown then return the exception object
                throw new Exception("CodedValueAttributeField '" + attributeField.ToString() + "' not available in the current PracticeSettingTypes context");
            }
        }

        /// <summary>
        /// Gets the value of a specified field in the CodedValueAttribute attribute
        /// on the given PracticeSettingTypes enumeration member.
        /// </summary>
        /// <param name="healthcareFacilityTypeCode">The enumeration member.</param>
        /// <param name="attributeField">Which attribute field to get the value of.</param>
        /// <returns>The value of the field.</returns>
        public static String GetPracticeSettingTypeCodeAttributeValue(PracticeSettingTypes healthcareFacilityTypeCode, CodedValueAttributeField attributeField)
        {
            String attributeValue = "";
            try
            {
                switch (attributeField)
                {
                    case CodedValueAttributeField.ConceptCode:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptCode);
                        break;

                    case CodedValueAttributeField.ConceptName:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptName);
                        break;

                    case CodedValueAttributeField.CodingSystem:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.CodingSystem);
                        break;

                    case CodedValueAttributeField.CodingSystemOID:
                        attributeValue = healthcareFacilityTypeCode.GetAttributeValue<CodedValueAttribute, string>(a => a.CodingSystemOID);
                        break;

                    default:
                        throw new CodedValueAttribException();
                }

                return attributeValue;
            }
            catch (InvalidOperationException)
            {
                //if an InvalidOperationException is thrown then return empty string
                return attributeValue;
            }
            catch (CodedValueAttribException)
            {
                //if an CodedValueAttribException is thrown then return the exception object
                throw new Exception("CodedValueAttributeField '" + attributeField.ToString() + "' not available in the current PracticeSettingTypes context");
            }
        }

        /// <summary>
        /// Inner exception class to handle invalid CodedValueAttrib used
        /// </summary>
        protected internal class CodedValueAttribException : Exception
        {
            /// <summary>
            /// Initialises a new instance of the <see cref="CodedValueAttribException"/> class.
            /// </summary>
            public CodedValueAttribException()
            {
            }
        }
    }
}