﻿using System;
using System.Runtime.Serialization;

namespace HIPS.ServiceContracts.Common
{
    /// <summary>
    /// Represents the purpose of an address. The valid codes are defined in AS 5017-2006
    /// in the table "Health Care Client Identifier Address Purpose", and the mappings to
    /// CDA in NEHTA eDischarge Summary CDA Implementation Guide v3.4.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum AddressPurpose
    {
        /// <summary>
        /// No value has been supplied. This represents an invalid invocation of the web service.
        /// If the address purpose has not been stated, service invokers should send "Unknown".
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents a business, workplace or office address. Maps to CDA address use code "WP".
        /// </summary>
        [EnumMember]
        Business = 1,

        /// <summary>
        /// Represents a mailing or postal address, used to send mail. Maps to CDA address use code "PST".
        /// </summary>
        [EnumMember]
        Mailing = 2,

        /// <summary>
        /// Represents a temporary address, which may be good for visit or mailing. Maps to CDA address use code "TMP".
        /// </summary>
        [EnumMember]
        Temporary = 3,

        /// <summary>
        /// Represents a residential or home address. Maps to CDA address use code "H".
        /// </summary>
        [EnumMember]
        Residential = 4,

        /// <summary>
        /// Represents an address whose purpose has not been stated, is unknown or inadequately
        /// described. In this case the CDA document will simply omit the Address Use Code.
        /// </summary>
        [EnumMember]
        Unknown = 9,
    }

    /// <summary>
    /// Represents the file type and semantics of an attachment.
    /// </summary>
    public enum AttachmentType
    {
        /// <summary>
        /// No value has been supplied. This represents an invalid invocation of the web service.
        /// </summary>
        None = 0,

        /// <summary>
        /// Represents an additional JPEG image attached to the document. This image will be shown
        /// in the CDA body under the main document. This file must be a JPEG image.
        /// </summary>
        AttachedImageJpeg = 1,

        /// <summary>
        /// Represents an additional PNG image attached to the document. This image will be shown
        /// in the CDA body under the main document. This file must be a PNG image.
        /// </summary>
        AttachedImagePng = 2,

        /// <summary>
        /// Represents an additional TIFF image attached to the document. This image will be shown
        /// in the CDA body under the main document. This file must be a TIFF image.
        /// </summary>
        AttachedImageTiff = 3,

        /// <summary>
        /// Represents an additional GIF image attached to the document. This image will be shown
        /// in the CDA body under the main document. This file must be a GIF image.
        /// </summary>
        AttachedImageGif = 4,

        /// <summary>
        /// Represents an additional linked document. This will be linked from the body of the
        /// CDA document. This file must be a PDF document.
        /// </summary>
        AttachedDocumentPdf = 5,

        /// <summary>
        /// Represents the organisational logo that will appear in the header of the CDA document.
        /// This file must be a PNG image. If no attachment is supplied with this type, and there
        /// is a logo in the database Hospital table, then HIPS will include the logo from the
        /// database.
        /// </summary>
        OrganisationalLogoPng = 6,
    }

    /// <summary>
    /// Represents an Australian state or territory. The valid codes are defined in AS 5017-2006
    /// in the table "Australian State/Territory Identifier - Postal", and the mappings to
    /// CDA are defined in NEHTA "eDischarge Summary CDA Implementation Guide v3.4" to use the
    /// same codes.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum AustralianState
    {
        /// <summary>
        /// No value has been supplied. This represents a valid invocation of the web service
        /// for an international address. For an Australian address, if the state or territory
        /// is unknown, service invokers should send "U".
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents New South Wales.
        /// </summary>
        [EnumMember]
        NSW = 1,

        /// <summary>
        /// Represents Victoria.
        /// </summary>
        [EnumMember]
        VIC = 2,

        /// <summary>
        /// Represents Queensland.
        /// </summary>
        [EnumMember]
        QLD = 3,

        /// <summary>
        /// Represents South Australia.
        /// </summary>
        [EnumMember]
        SA = 4,

        /// <summary>
        /// Represents Western Australia.
        /// </summary>
        [EnumMember]
        WA = 5,

        /// <summary>
        /// Represents Tasmania.
        /// </summary>
        [EnumMember]
        TAS = 6,

        /// <summary>
        /// Represents Northern Territory.
        /// </summary>
        [EnumMember]
        NT = 7,

        /// <summary>
        /// Represents Australian Capital Territory.
        /// </summary>
        [EnumMember]
        ACT = 8,

        /// <summary>
        /// Represents an Australian address whose state or territory is unknown.
        /// </summary>
        [EnumMember]
        U = 9,
    }

    /// <summary>
    /// Represents the format of a CDA package.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum CdaPackageFormat
    {
        /// <summary>
        /// No format has been defined.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// The package format is XDM-ZIP.
        /// </summary>
        [EnumMember]
        XdmZip = 1,

        /// <summary>
        /// The package format is HL7-MDM.
        /// </summary>
        [EnumMember]
        Hl7Mdm = 2,
    }

    /// <summary>
    /// Represents the medium of an electronic communication detail. The valid
    /// codes are defined in AS 5017-2006 in the table "Health Care Client
    /// Electronic Communication Medium", and the mappings to HL7v3 (CDA) URL
    /// schemes in NEHTA "eDischarge Summary CDA Implementation Guide v3.4".
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum ElectronicCommunicationMedium
    {
        /// <summary>
        /// No value has been supplied. This represents an invalid invocation of the web service.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents a fixed telephone. Maps to URL scheme "tel".
        /// </summary>
        [EnumMember]
        FixedTelephone = 1,

        /// <summary>
        /// Represents a mobile telephone. Maps to URL scheme "tel" with use code "MC".
        /// </summary>
        [EnumMember]
        Mobile = 2,

        /// <summary>
        /// Represents a facsimile machine. Maps to URL scheme "fax".
        /// </summary>
        [EnumMember]
        Fax = 3,

        /// <summary>
        /// Represents a pager. Maps to URL scheme "tel" with use code "PG".
        /// </summary>
        [EnumMember]
        Pager = 4,

        /// <summary>
        /// Represents email. Maps to URL scheme "mailto".
        /// </summary>
        [EnumMember]
        Email = 5,

        /// <summary>
        /// Represents a URL. The URL scheme of the URL in the Detail element is used without mapping.
        /// </summary>
        [EnumMember]
        URL = 6,
    }

    /// <summary>
    /// Represents the usage of an electronic communication detail. The valid
    /// codes are defined in AS 5017-2006 in the table "Health Care Client
    /// Electronic Communication Usage Code", and the mappings to HL7v3 (CDA)
    /// Telecommunication Address Use (TAU) codes in NEHTA "eDischarge Summary
    /// CDA Implementation Guide v3.4".
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum ElectronicCommunicationUsage
    {
        /// <summary>
        /// No value has been supplied. This represents a valid invocation of the web service because the usage code is optional.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents a communication address for business use. Maps to TAU code "WP".
        /// </summary>
        [EnumMember]
        Business = 1,

        /// <summary>
        /// Represents a communication address for personal use. Maps to TAU code "H".
        /// </summary>
        [EnumMember]
        Personal = 2,

        /// <summary>
        /// Represents a  communication address for both business and personal use. Maps to TAU code "WP H".
        /// </summary>
        [EnumMember]
        Both = 3,
    }

    /// <summary>
    /// This is used to specify the conditions when the PCEHR
    /// advertised status should be refreshed.
    /// </summary>
    public enum ForceRefreshOption
    {
        /// <summary>
        /// Indicates that a value has not been specified
        /// providing this value will cause a validation error.
        /// </summary>
        None = 0,

        /// <summary>
        /// Indicates that there is no need to refresh the PCEHR
        /// advertised status and the previously stored information will be returned.
        /// </summary>
        Never = 1,

        /// <summary>
        /// Indicates that the PCEHR advertised status should be refreshed
        /// if there is no previously stored information that indicates the patient has a PCEHR.
        /// </summary>
        WhenNotAdvertised = 2,

        /// <summary>
        /// Indicates that the PCEHR advertised status should always be refreshed.
        /// </summary>
        Always = 3
    }

    /// <summary>
    /// Represents an IHI record status.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum IhiRecordStatus
    {
        /// <summary>
        /// No value has been supplied. Where a value for IhiRecordStatus is required, this
        /// represents an invalid invocation of the web service.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents verified record status.
        /// </summary>
        [EnumMember]
        Verified = 1,

        /// <summary>
        /// Represents unverified record status.
        /// </summary>
        [EnumMember]
        Unverified = 2,

        /// <summary>
        /// Represents provisional record status.
        /// </summary>
        [EnumMember]
        Provisional = 3,

        /// <summary>
        /// Represents unknown record status. This status is assigned to patients whose IHI
        /// has not yet been obtained.
        /// </summary>
        [EnumMember]
        Unknown = 4,
    }

    /// <summary>
    /// Represents an IHI status. Some values are defined by the national HI Service.
    /// Others are defined by HIPS to represent various unresolved alert conditions.
    /// See the documentation for each member for details.
    /// </summary>
    [Serializable]
    public enum IhiStatus
    {
        /// <summary>
        /// No value has been supplied. Where a value for IhiStatus is required, this
        /// represents an invalid invocation of the web service.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents active status as defined by the HI Service.
        /// </summary>
        [EnumMember]
        Active = 1,

        /// <summary>
        /// Represents deceased status as defined by the HI Service.
        /// </summary>
        [EnumMember]
        Deceased = 2,

        /// <summary>
        /// Represents retired status as defined by the HI Service.
        /// </summary>
        [EnumMember]
        Retired = 3,

        /// <summary>
        /// Represents expired status as defined by the HI Service.
        /// </summary>
        [EnumMember]
        Expired = 4,

        /// <summary>
        /// Represents resolved status as defined by the HI Service. The current
        /// IHI for a patient should never have this status, but it can be
        /// recorded for historical IHI assignments.
        /// </summary>
        [EnumMember]
        Resolved = 5,

        /// <summary>
        /// Represents unknown status. This status is assigned to patients whose IHI
        /// has not yet been obtained, or when an IHI of record status Unverified or
        /// Provisional was found.
        /// </summary>
        [EnumMember]
        Unknown = 6,

        /// <summary>
        /// Represents an unresolved alert due to obtaining an IHI when the same
        /// IHI is assigned to another patient record from the same hospital. The
        /// administrator must determine whether the two records represent the
        /// same person, if so then merge the two records, if not then report a
        /// potential replica to Medicare Australia.
        /// </summary>
        [EnumMember]
        DuplicateIhi = 100,

        /// <summary>
        /// Represents an unresolved alert due to obtaining an IHI using the same
        /// demographic details that are assigned to another patient record from
        /// the same hospital. The administrator must determine whether the two
        /// records represent the same person, if so then merge the two records,
        /// if not then report a potential replica to Medicare Australia.
        /// </summary>
        [EnumMember]
        DuplicatePatient = 101,

        /// <summary>
        /// Represents an unresolved alert due to merging two patient records that have
        /// different IHI numbers. The administrator must choose the correct IHI to
        /// resolve this alert.
        /// </summary>
        [EnumMember]
        MergeConflict = 102,

        /// <summary>
        /// Represents a patient whose IHI could not be obtained or validated because of
        /// an error while connecting to the HI Service. The IHI search will be repeated
        /// by the background process until it is successful.
        /// </summary>
        [EnumMember]
        ServiceUnavailable = 103,

        /// <summary>
        /// Represents an unresolved alert due to changing the Medicare card number or
        /// DVA file number of the patient, if the patient already had an IHI assigned,
        /// and a search using the new Medicare or DVA number found no IHI or found a
        /// different IHI to the one that was assigned to the patient.
        /// </summary>
        [EnumMember]
        MedicareDvaChangeMismatch = 104,

        /// <summary>
        /// Represents an unresolved alert due to changing the date of birth of the patient,
        /// if the patient already had an IHI assigned, and a search using the current IHI
        /// with the new date of birth found no IHI.
        /// </summary>
        [EnumMember]
        DemographicMismatch = 105
    }

    /// <summary>
    /// Represents the level of a message returned by a service operation.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum MessageLevel
    {
        /// <summary>
        /// No level has been defined.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// The message contains information.
        /// </summary>
        [EnumMember]
        Information = 1,

        /// <summary>
        /// The message contains a warning.
        /// </summary>
        [EnumMember]
        Warning = 2,

        /// <summary>
        /// The message contains an error description.
        /// </summary>
        [EnumMember]
        Error = 3
    }

    /// <summary>
    /// Represents the status of the patient at the time of discharge and the
    /// place to which the person was or will be released. The valid codes
    /// are defined in AIHW 270094 titled "Episode of admitted patient care-
    /// separation mode, code N". Shorter, more user friendly display names
    /// were developed by NEHTA.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum ModeOfSeparation
    {
        /// <summary>
        /// No value has been supplied. This represents an invalid invocation of the web service.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents a discharge or transfer to an acute hospital.
        /// </summary>
        [EnumMember]
        AcuteHospital = 1,

        /// <summary>
        /// Represents a discharge or transfer to a residential aged care service, unless this is the usual place of residence.
        /// </summary>
        [EnumMember]
        AgedCareService = 2,

        /// <summary>
        /// Represents a discharge or transfer to a psychiatric hospital.
        /// </summary>
        [EnumMember]
        PsychiatricCare = 3,

        /// <summary>
        /// Represents a discharge or transfer to other health care accommodation (includes mothercraft hospitals)
        /// </summary>
        [EnumMember]
        OtherHealthService = 4,

        /// <summary>
        /// Represents a statistical discharge - type change.
        /// </summary>
        [EnumMember]
        AdministrativeDischarge = 5,

        /// <summary>
        /// Represents that the patient left against medical advice or was discharged at own risk.
        /// </summary>
        [EnumMember]
        SelfDischarge = 6,

        /// <summary>
        /// Represents a statistical discharge from leave.
        /// </summary>
        [EnumMember]
        AdministrativeFromLeave = 7,

        /// <summary>
        /// Represents that the patient died.
        /// </summary>
        [EnumMember]
        Deceased = 8,

        /// <summary>
        /// Represents that the patient was discharged to his/her usual residence, own accommodation or a welfare institution (includes prisons, hostels and group homes providing primarily welfare services).
        /// </summary>
        [EnumMember]
        Home = 9,
    }

    /// <summary>
    /// Represents the PCEHR participation status of a patient.
    /// </summary>
    public enum ParticipationStatus
    {
        /// <summary>
        /// HIPS was not able to obtain or validate the IHI for this patient. HIPS will not be able to
        /// upload the discharge summary for this patient until this situation is resolved.
        /// </summary>
        NoValidIhi,

        /// <summary>
        /// This patient may not have registered for PCEHR, or may have chosen to hide the existence of
        /// his/her PCEHR. The clinical system should not send the discharge summary, unless the patient
        /// explicitly requests the provider to do so.
        /// </summary>
        PcehrNotAdvertised,

        /// <summary>
        /// This patient has chosen to advertise the existence of his/her PCEHR, or the health provider
        /// organisation has already gained access to the PCEHR, so the clinical system should send the
        /// discharge summary.
        /// </summary>
        PcehrAdvertised,

        /// <summary>
        /// This patient disclosed the existence of his/her PCEHR to the health provider organisation, and
        /// thereby requested the upload of his/her discharge summary for all hospitals that are part of
        /// this health provider organisation, so the clinical system should send the discharge summary,
        /// regardless of whether the PCEHR is advertised.
        /// </summary>
        RequestedUpload
    }

    /// <summary>
    /// Represents the outcome of a service operation.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum ResponseStatus
    {
        /// <summary>
        /// No status has been defined.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// The service operation was successful.
        /// </summary>
        [EnumMember]
        OK = 1,

        /// <summary>
        /// The service operation may have been successful, with associated warning messages.
        /// </summary>
        [EnumMember]
        Warning = 2
    }

    /// <summary>
    /// Represents the sex of a person. The valid codes are defined by AS 5017-2006
    /// in table "Health Care Client Identifier Sex".
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum Sex
    {
        /// <summary>
        /// No value has been supplied. Where a value for Sex is required, this
        /// represents an invalid invocation of the web service. If the IHI
        /// search was successful with the Sex not stated, send the value
        /// "NotStatedOrInadequatelyDescribed".
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents male. Mapped to CDA code "M".
        /// </summary>
        [EnumMember]
        Male = 1,

        /// <summary>
        /// Represents female. Mapped to CDA code "F".
        /// </summary>
        [EnumMember]
        Female = 2,

        /// <summary>
        /// Represents intersex or indeterminate. Mapped to CDA code "I".
        /// </summary>
        [EnumMember]
        IntersexOrIndeterminate = 3,

        /// <summary>
        /// Represents not stated or inadequately described. Mapped to CDA code "N".
        /// </summary>
        [EnumMember]
        NotStatedOrInadequatelyDescribed = 4,
    }

    /// <summary>
    /// Represents the status of this version of the document as communicated
    /// by the source system. The valid codes are defined by NCTIS titled
    /// "Admin Codes - Document Status".
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum SourceDocumentStatus
    {
        /// <summary>
        /// No value has been supplied. This represents an invalid invocation of the web service.
        /// </summary>
        [EnumMember]
        None = 0,

        /// <summary>
        /// Represents an interim or draft version of the document. Interim versions are not normally
        /// uploaded to the PCEHR system.
        /// </summary>
        [EnumMember]
        Interim = 1,

        /// <summary>
        /// Represents a final version of the document. This may be the first final version or an amendment.
        /// </summary>
        [EnumMember]
        Final = 2,

        /// <summary>
        /// Represents a notification that the document has been withdrawn by the authoring organisation.
        /// Withdrawn versions are intended for delivery to point-to-point (P2P) recipients of the document
        /// and are not normally uploaded to the PCEHR system. Instead the document can be removed from the
        /// PCEHR system with a separate web service invocation.
        /// </summary>
        [EnumMember]
        Withdrawn = 3,
    }

    /// <summary>
    /// Status used for the mapping of the returned type from the HPI-I status.
    /// </summary>
    [DataContract(Namespace = HIPS.ServiceContracts.Common.Constants.ServiceNamespace.COMMON_V1)]
    public enum HpiiStatus
    {
        /// <summary>
        /// Undefined, this is the default value if the enum is left unset.
        /// The validation engine uses this to test and assert that the enum has been set (if required)
        /// and is therefore valid.
        /// </summary>
        [EnumMember]
        Undefined = 0,

        /// <summary>
        /// Active HpiiStatus Record
        /// </summary>
        [EnumMember]
        Active = 1,

        /// <summary>
        /// Deactivated HpiiStatus Record
        /// </summary>
        [EnumMember]
        Deactivated = 2,

        /// <summary>
        /// Retired HpiiStatus Record
        /// </summary>
        [EnumMember]
        Retired = 3
    }
}