﻿using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using HIPS.PcehrDataStore.Schemas;
using Ionic.Zip;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.CdaPackaging.XdmZipRepresentation
{
    /// <summary>
    /// Conformance Test Specification: CDA Packaging
    ///                                 Version 1.4 — 12 September 2012
    /// Profile Name:	XDM-ZIP Representation
    /// Test Case ID:   DEXS-L_154
    /// Objective:
    /// To check that the XDM Zip CDA Package only contains the root part, the signature and packaged attachments.
    /// </summary>
    [TestClass]
    public class XdmZipRepresentation1A : CcaTest
    {
        private ICollection<ZipEntry> entries;
        private ClinicalDocumentVersion version;
        private CdaDocument uploadedDocument;

        /// <summary>
        /// Shared helper method to ensure that a document has been uploaded
        /// and get the package, uploaded document and ZIP entries.
        /// The document will be reused between the tests in this class.
        /// </summary>
        private void UploadDocument()
        {
            SharedPackage shared = SharedPackage.GetSharedInstance1A(this, SampleDocumentType.DischargeSummary1A);
            this.version = shared.Version;
            this.uploadedDocument = shared.CdaDocument;
            this.entries = shared.PackageEntries;
        }

        /// <summary>
        /// (Note: This test is only relevant in the context of PCEHR)
        /// Verify that the only files contained in the CDA Package zip file
        /// are CDA_ROOT.XML, CDA_SIGN.XML and packaged attachment files.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_DEXS_L_154()
        {
            UploadDocument();
            List<ZipEntry> extraFiles = entries.Where(a => !a.IsDirectory
                && a.FileName != "IHE_XDM/SUBSET01/CDA_ROOT.XML"
                && a.FileName != "IHE_XDM/SUBSET01/CDA_SIGN.XML"
                && !Regex.IsMatch(a.FileName.ToUpper(), "^IHE_XDM/SUBSET01/[^/]+.(PNG|JPG|JPEG|PDF|TIF|TIFF)$")).ToList();

            LogAssert.AreEqual(0, extraFiles.Count, DialogueResource.NumberOfExtraFiles);
        }

        /// <summary>
        /// (Note: This test is only relevant in the context of PCEHR)
        /// Verify that the CDA Package zip file does not contain INDEX.HTM,
        /// README.TXT or Repository Metadata (e.g. METADATA.XML)
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_DEXS_L_155()
        {
            UploadDocument();
            List<ZipEntry> index = entries.Where(a => !a.IsDirectory && a.FileName.ToUpper().EndsWith("/INDEX.HTM")).ToList();
            List<ZipEntry> readme = entries.Where(a => !a.IsDirectory && a.FileName.ToUpper().EndsWith("/README.TXT")).ToList();
            List<ZipEntry> metadata = entries.Where(a => !a.IsDirectory && a.FileName.ToUpper().EndsWith("/METADATA.XML")).ToList();

            LogAssert.AreEqual(0, index.Count, DialogueResource.NumberOfFilesNamedIndexHtm);
            LogAssert.AreEqual(0, readme.Count, DialogueResource.NumberOfFilesNamedReadmeTxt);
            LogAssert.AreEqual(0, metadata.Count, DialogueResource.NumberOfFilesNamedMetadataXml);
        }

        /// <summary>
        /// (Note: This test is only relevant in the context of PCEHR)
        /// If one or more packaged attachments are present:
        /// a) Verify that the packaged attachment files are not, themselves,
        ///    CDA Package or CDA document files.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_DEXS_L_156_A()
        {
            UploadDocument();
            List<ZipEntry> attachments = entries.Where(a =>
                    !a.IsDirectory
                 && !a.FileName.EndsWith("CDA_ROOT.XML")
                 && !a.FileName.EndsWith("CDA_SIGN.XML")).ToList();

            if (attachments.Count == 0)
            {
                LogAssert.Inconclusive("There were no packaged attachments");
            }

            foreach (ZipEntry entry in attachments)
            {
                LogAssert.IsFalse(entry.FileName.EndsWith(".XML"), "Attachment is not an XML file", "Attachment is an XML file");
                LogAssert.IsFalse(entry.FileName.EndsWith(".ZIP"), "Attachment is not a ZIP file", "Attachment is a ZIP file");
            }
        }

        /// <summary>
        /// Verify that there is exactly one submission set.
        ///
        /// The term submission set is defined in [XDM2006] Cross-Enterprise Document Media Interchange (XDM).
        ///
        /// This document states that:
        /// The IHE_XDM directory shall contain one sub-directory per submission set included on the media.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XDM_ZIP_106()
        {
            UploadDocument();

            // The submission sets are the subdirectories directly under the IHE_XDM directory.
            // Their name consists of "IHE_XDM/" followed by one or more non-slash chars followed by a slash.
            var submissionSets = entries.Where(a => a.IsDirectory && Regex.IsMatch(a.FileName, "^IHE_XDM/[^/]+/$"));
            Log("The submission sets are: {0}", string.Join(", ", submissionSets.Select(a => a.FileName.Split('/')[1])));

            // Verify that there is exactly one submission set
            LogAssert.AreEqual(1, submissionSets.Count(), DialogueResource.NumberOfSubmissionSets);
        }

        /// <summary>
        /// Verify that the root has the fixed filename "CDA_ROOT.XML".
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XDM_ZIP_108()
        {
            UploadDocument();

            ZipEntry signatureFile = entries.FirstOrDefault(a => a.FileName == "IHE_XDM/SUBSET01/CDA_ROOT.XML");
            LogAssert.IsTrue(signatureFile != null, "Package contains CDA_ROOT.XML file", "Package does not contain CDA_ROOT.XML file");

            CdaDocument doc = new CdaDocument(CdaPackage.ExtractDocument(version.Package));
            LogAssert.IsTrue(doc.Validate(this), "CDA_ROOT.XML file validates as CDA", "CDA_ROOT.XML does not validate as CDA");
        }

        /// <summary>
        /// If an eSignature is present in the CDA Package:
        /// a) Verify that the eSignature has the fixed filename "CDA_SIGN.XML"
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XDM_ZIP_109_A()
        {
            UploadDocument();
            ZipEntry signatureFile = entries.FirstOrDefault(a => a.FileName == "IHE_XDM/SUBSET01/CDA_SIGN.XML");
            LogAssert.IsTrue(signatureFile != null, "Package contains CDA_SIGN.XML file", "Package does not contain CDA_SIGN.XML file");

            // Verify that the CDA_SIGN.XML file has the expected structure for an eSignature
            byte[] eSignatureBinary = CdaPackage.ExtractSignature(version.Package);
            CdaSignature eSignature = new CdaSignature(eSignatureBinary, this);
            LogAssert.IsTrue(eSignature.ValidateCdaSignature(), "eSignature was valid", "eSignature was invalid");
        }
    }
}