﻿using System.IO;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentConsumption
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  doesPCEHRExist
    /// </summary>
    [TestClass]
    public class NOC_GainPcehrAccess : CcaTest
    {
        /// <summary>
        /// Test No:         22
        /// Objective/Input:
        ///
        /// Ensure a valid IHI number format and healthcare provider UserID are
        /// used to gain access to a PCEHR.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR system.
        /// Confirmation that the payload is well formed and complies with the
        /// gainPCEHRAccess Request XML schema.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_022()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            GainPcehrAccessResponse pcehrResponse = ProxyHelper.PcehrProxy.GainAccessWithOutCode(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GainAccessWithoutCode, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            VerifyGainAccessPayload(audit);
        }

        /// <summary>
        /// Test No:         23
        /// Objective/Input:
        /// Ensure that a client system CIS or CSP can prevent gainPCEHRAccess
        /// request when invalid IHI number is used.
        ///
        /// Expected Result:
        /// The request should not be transmitted to the PCEHR System. Client
        /// system should indicate to the user when the IHI number of the
        /// individual is invalid.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_023()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false);
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithOutCode(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);

            // Check that the access was prevented because the IHI was invalid
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.IsTrue(patient.TargetPatientMasterId != pcehrAudit.PatientMasterId,
                "The request was not transmitted to the PCEHR System",
                "The request was transmitted to the PCEHR System");
        }

        /// <summary>
        /// Test No:         24
        /// Objective/Input:
        ///
        /// Ensure a valid IHI number format is used to gain access to a PCEHR where the accessType is Emergency.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR system.
        /// Confirmation that the payload is well formed and complies with the
        /// gainPCEHRAccess Request XML schema.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_024()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            GainPcehrAccessResponse pcehrResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GainEmergencyAccess, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            VerifyGainAccessPayload(audit);
        }

        /// <summary>
        /// Test No:         25
        /// Objective/Input:
        ///
        /// To ensure that the client system CIS or CSP can execute
        /// gainPCEHRAccess with an Access Code correctly.
        /// Verify that when a verified IHI number format is used to gain
        /// access to a PCEHR and authorisation details element is set to
        /// AccessCode with access code element populated.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Confirmation that the payload is well formed and created according
        /// to the gainPCEHRAccess Request XML schema.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_025()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            GainPcehrAccessResponse pcehrResponse = ProxyHelper.PcehrProxy.GainAccessWithCode(identifier, patient.TestPatientRecordAccessCode, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GainAccessWithCode, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            VerifyGainAccessPayload(audit, checkAccessCode: true);
        }

        private void VerifyGainAccessPayload(PcehrAudit audit, bool checkAccessCode = false)
        {
            // Confirm that the payload is well formed and complies with the gainPCEHRAccess Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/PCEHRProfile/1.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
            XmlNode payload = doc.SelectSingleNode("/s:Envelope/s:Body/p:gainPCEHRAccess", nsmgr);
            doc.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("p"), "PCEHR_GainPCEHRAccess.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("c"), "PCEHR_CommonTypes.xsd");
            try
            {
                doc.Validate(null, payload);
                doc.Validate(null);
                LogAssert.IsTrue(true, "Validates against SOAP, GainPCEHRAccess and CommonTypes schemas", "no");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }

            // Check that the patient ID was provided in the request
            XmlNode ihiNumberNode = doc.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:ihiNumber", nsmgr);
            LogAssert.AreEqual(patient.IhiInformation.Ihi, ihiNumberNode.InnerText, "IHINumber element value");

            // Check that the user ID was supplied
            XmlNode userIdNode = doc.SelectSingleNode("/s:Envelope/s:Header/c:PCEHRHeader/c:User/c:ID", nsmgr);
            LogAssert.AreEqual(patient.GetTestUser().Login, userIdNode.InnerText, "User ID element value");

            if (checkAccessCode)
            {
                XmlNode accessCodeNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:gainPCEHRAccess/p:PCEHRRecord/p:authorisationDetails/p:accessCode", nsmgr);
                LogAssert.IsNotNull(accessCodeNode, "Access code element");
            }
        }

        /// <summary>
        /// Test No:         26
        /// Objective/Input:
        /// To show the ability of the client system to prevent non-conformant
        /// gainPCEHRAccess Request against RCAS-T23. Verify that when an IHI
        /// number is used and an Access Code is used for the accessType
        /// element, and the code element is not supplied the request is not
        /// transmitted to the PCEHR System.
        ///
        /// Expected Result:
        /// The request should not be transmitted to the PCEHR System. Client
        /// system should indicate to the user when Access Type is set with the
        /// 'AccessCode', the Access Code MUST be supplied.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_026()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            GainPcehrAccessResponse pcehrResponse = ProxyHelper.PcehrProxy.GainAccessWithCode(identifier, "", user);
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidAccessCode, pcehrResponse.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that the access was prevented because an attempt was made to gain access with code but no code was supplied.
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.IsTrue(patient.TargetPatientMasterId != pcehrAudit.PatientMasterId,
                "The request was not transmitted to the PCEHR System",
                "The request was transmitted to the PCEHR System");
        }

        /// <summary>
        /// Test No:         27
        /// Objective/Input:
        /// To show the ability of the client system to prevent non-conformant
        /// gainPCEHRAccess Request against RCAS-T 30. Verify that when an
        /// attempt to execute gainPCEHRAccess request without an IHI number is
        /// not permitted.
        ///
        /// Expected Result:
        /// The request should not be transmitted to the PCEHR System. Client
        /// system should indicate to the user when the IHI number of the
        /// individual who owns the PCEHR is not used and that it MUST be
        /// supplied.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_027()
        {
            patient = CcaPatient.GetPatientWithoutIhi();
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithOutCode(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);

            // Check that the access was prevented because there was no IHI for the patient
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.IsTrue(patient.TargetPatientMasterId != pcehrAudit.PatientMasterId,
                "The request was not transmitted to the PCEHR System",
                "The request was transmitted to the PCEHR System");
        }

        /// <summary>
        /// Test No:         28
        /// Objective/Input:
        ///
        /// To show that the client system can handle a functional error
        /// PCEHR_ERROR_5101 - PCEHR not found
        /// when a PCEHR is not found when a gainPCEHRAccess operation is performed.
        ///
        /// Expected Result:
        /// The request is successfully transmitted to the PCEHR System and
        /// Functional Error PCEHR_ERROR_5101 is returned. Client system is
        /// able to handle and inform the user that a PCEHR was not found.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_028()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 37");
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithOutCode(
                patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);
            LogAssert.AreEqual("PCEHR_ERROR_5101", response.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("eHealth Record not found", response.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);

            // Check that access was not granted.
            LogAssert.AreEqual(GainPcehrAccessStatus.Deny, response.AccessPermission, "Access permission");
        }

        /// <summary>
        /// Test No:         29
        /// Objective/Input:
        ///
        /// To show that the client system can handle a functional error
        /// PCEHR_ERROR_5102 - PCEHR is found but access code is required
        /// when a PCEHR is found but access code is required when a
        /// gainPCEHRAccess operation is performed.
        ///
        /// Expected Result:
        /// The request is successfully transmitted to the PCEHR System and
        /// functional error PCEHR_ERROR_5102 is returned. Client system is
        /// able to handle and inform the user that a PCEHR was found but
        /// an access code is required.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_029()
        {
            Log("For this test the HPI-O must not have gained access already. Switching to HPI-O 2 will achieve this.");
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            HpiOHelper.SwitchToHpiO(patient, this, 2);
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithOutCode(
                patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);
            LogAssert.AreEqual("PCEHR_ERROR_5102", response.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("eHealth Record is found but Access Code is required", response.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);

            // Check that access was not granted.
            LogAssert.AreEqual(GainPcehrAccessStatus.Deny, response.AccessPermission, "Access permission");
            HpiOHelper.SwitchToHpiO(patient, this, 1);
        }

        /// <summary>
        /// Test No:         30
        /// Objective/Input:
        ///
        /// To show that the client system can handle a Standard Error (PCEHR_ERROR_0505 – Invalid HPI-O)
        /// The user performs the gain PCEHR access operation with an invalid HPI-O. The client system is
        /// configured with an invalid HPI-O.
        ///
        /// Expected Result:
        ///
        /// The request is successfully transmitted to the PCEHR System. The client system is able to handle
        /// the Standard error (PCEHR_ERROR_0505 - Invalid HPI-O) and informs the user appropriately.
        /// </summary>
        [TestMethod]
        public void PCEHR_NOC_030()
        {
            // Create the patient record
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: false);

            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithOutCode(
                patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);
            if (patient.TargetHospital.PcehrCsp)
            {
                LogAssert.AreEqual("notAuthorised", response.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
                LogAssert.AreEqual("PCEHR_ERROR_0004 - Authorisation denied", response.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
            }
            else
            {
                LogAssert.AreEqual("badParam", response.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
                LogAssert.AreEqual("PCEHR_ERROR_0505 - Invalid HPI-O", response.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
            }
            HpiOHelper.ChangeHpiO(patient, this, makeValid: true);
        }

        /// <summary>
        /// Test No:         31
        /// Objective/Input:
        ///
        /// To show that the client system can handle a functional error when a
        /// gainPCEHRAccess operation is performed with an invalid AccessCode
        /// where a HPI-O requires an access code to access the PCEHR for the
        /// IHI. The client system uses an invalid access code.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. The
        /// client system is able to handle the PCEHR_ERROR_5103 error in the
        /// response.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_031()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(
                patient.TargetPatientIdentifier, "The Wrong Code", patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);
            LogAssert.AreEqual("PCEHR_ERROR_5103", response.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("eHealth Record is found but Access Code is invalid", response.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);

            // Check that access was not granted.
            LogAssert.AreEqual(GainPcehrAccessStatus.Deny, response.AccessPermission, "Access permission");
        }
    }
}