﻿/*
 * Copyright 2013 NEHTA
 *
 * Licensed under the NEHTA Open Source (Apache) License; you may not use this
 * file except in compliance with the License. A copy of the License is in the
 * 'license.txt' file, which should be provided with this work.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using JetBrains.Annotations;
using Nehta.HL7.CDA;
using Nehta.VendorLibrary.CDA.Common;
using Nehta.VendorLibrary.CDA.Common.Enums;
using Nehta.VendorLibrary.Common;

namespace Nehta.VendorLibrary.CDA.SCSModel.Common
{
    /// <summary>
    /// This class is designed to encapsulate the properties within a CDA document that make up 
    /// an imaging examination result.
    /// </summary>
    [Serializable]
    [DataContract]
    [KnownType(typeof(CodableText))]
    [KnownType(typeof(ResultGroup))]
    [KnownType(typeof(Request))]
    [KnownType(typeof(DischargeSummary.ImagingExaminationResult))]
    public class ImagingExaminationResult : IImagingExaminationResult
    {
        #region Properties
        /// <summary>
        /// Examination result name
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ICodableText ExaminationResultName { get; set; }

        /// <summary>
        /// Modality
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ICodableText Modality { get; set; }

        /// <summary>
        /// Anatomical site
        /// </summary>
        [CanBeNull]
        [DataMember]
        public List<AnatomicalSite> AnatomicalSite { get; set; }

        /// <summary>
        /// Result status
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ResultStatus? ExaminationResultStatus { get; set; }

        /// <summary>
        /// Clinical information provided
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String ClinicalInformationProvided { get; set; }

        /// <summary>
        /// Findings
        /// </summary>
        [CanBeNull]
        [DataMember]
        public String Findings { get; set; }

        /// <summary>
        /// A list of imaging results
        /// </summary>
        [CanBeNull]
        [DataMember]
        public List<IImagingResultGroup> ResultGroup { get; set; }

        /// <summary>
        /// examination result representation
        /// </summary>
        [CanBeNull]
        [DataMember]
        public string ExaminationResultRepresentation { get; set; }

        /// <summary>
        /// A list of examination requests
        /// </summary>
        [CanBeNull]
        [DataMember]
        public List<IImagingExaminationRequest> ExaminationRequestDetails { get; set; }

        /// <summary>
        /// Examination procedures
        /// </summary>
        [CanBeNull]
        [DataMember]
        public List<String> ExaminationProcedures { get; set; }

        /// <summary>
        /// The date / time of the result
        /// </summary>
        [CanBeNull]
        [DataMember]
        public ISO8601DateTime ResultDateTime { get; set; }

        /// <summary>
        /// Provide a custom Narrative 
        /// </summary>
        [CanBeNull]
        [DataMember]
        public StrucDocText CustomNarrativeImagingExaminationResult { get; set; }

        #endregion

        #region Constructors
        internal ImagingExaminationResult()
        {
        }
        #endregion

        #region Validation
        /// <summary>
        /// Validates this imaging examination result
        /// </summary>
        /// <param name="path">The path to this object as a string</param>
        /// <param name="messages">the validation messages to date, these may be added to within this method</param>
        public void Validate(string path, List<ValidationMessage> messages)
        {
            var vb = new ValidationBuilder(path, messages);

            if (vb.ArgumentRequiredCheck("ExaminationResultName",ExaminationResultName))
            {
                if (ExaminationResultName != null) ExaminationResultName.ValidateMandatory(vb.Path + "ExaminationResultName", messages);
            }

            vb.ArgumentRequiredCheck("OverallResultStatus", ExaminationResultStatus);
            vb.ArgumentRequiredCheck("ResultDateTime", ResultDateTime);

            if (Modality != null)
            {
                Modality.Validate(vb.Path + "Modality", messages);
            }

            if(AnatomicalSite != null && AnatomicalSite.Any())
            {
                AnatomicalSite.ForEach(anatomicalSite => anatomicalSite.Validate(vb.Path + "AnatomicalSite", messages));
            }

            if (ExaminationRequestDetails != null)
            {
                ExaminationRequestDetails.ForEach(ExaminationRequestDetail => ExaminationRequestDetail.Validate(vb.Path + "ExaminationRequestDetails", messages));
            }

            if (ResultGroup != null && ResultGroup.Any())
            {
                ResultGroup.ForEach(imagingResultGroup => imagingResultGroup.Validate(vb.Path + "ResultGroup", messages));
            }

            if(ExaminationProcedures != null && ExaminationProcedures.Any())
            {
                ExaminationProcedures.ForEach(procedure => vb.ArgumentRequiredCheck(vb.Path + "ExaminationProcedure", procedure));
            }
        }
        #endregion
    }
}
